<?php

/**
 * Class Handler
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2017 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Lifestyle\Sylius\RestApiClientBundle\Services\ProductOption\Version1\Create;

use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\ClientException;
use JMS\Serializer\SerializerInterface;
use Lifestyle\DataCollector\DataCollectorInterface;
use Lifestyle\Sylius\RestApiClientBundle\Exception\RestRequestFailedException;
use Psr\Log\LoggerInterface;

/**
 * Class Handler
 * @package Lifestyle\Sylius\RestApiClientBundle\Services\ProductOption\Version1\Create
 */
class Handler
{
    const API_CREATE_URI = '/api/v1/product-options/';
    const CLIENT_EXCEPTION_MESSAGE = 'Could not add product option object!';

    /**
     * @var SerializerInterface
     */
    private $serializer;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * @var ClientInterface
     */
    private $client;

    /**
     * @var Mapper
     */
    private $mapper;

    /**
     * Handler constructor.
     * @param SerializerInterface $serializer
     * @param LoggerInterface $logger
     * @param ClientInterface $client
     * @param Mapper $mapper
     */
    public function __construct(
        SerializerInterface $serializer,
        LoggerInterface $logger,
        ClientInterface $client,
        Mapper $mapper
    ) {
        $this->serializer = $serializer;
        $this->logger = $logger;
        $this->client = $client;
        $this->mapper = $mapper;
    }

    /**
     * @param DataCollectorInterface $collector
     * @param string $optionCode
     * @return bool
     * @throws RestRequestFailedException
     * @throws \GuzzleHttp\Exception\GuzzleException
     */
    public function create(DataCollectorInterface $collector, string $optionCode)
    {
        $productOptionRequest = $this->mapper->mapRequestModel($collector, $optionCode);

        $options = [
            'body' => $this->serializer->serialize($productOptionRequest, 'json'),
            'headers' => ['Content-type' => 'application/json']
        ];
        $uri = static::API_CREATE_URI;

        $this->logger->info(sprintf('Send POST request %s with body: %s', $uri, $options['body']));

        try {
            $response = $this->client->request('POST', $uri, $options);
        } catch (ClientException $exception) {
            throw new RestRequestFailedException(
                self::CLIENT_EXCEPTION_MESSAGE . ' - ' . $exception->getResponse()->getBody()->getContents(),
                $exception->getCode()
            );
        } catch (\Exception $exception) {
            throw new RestRequestFailedException(
                self::CLIENT_EXCEPTION_MESSAGE . ' - ' . $exception->getMessage(),
                $exception->getCode()
            );
        }
        return 204 === $response->getStatusCode();
    }
}
