<?php

/**
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2018 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Lifestyle\Sylius\RestApiClientBundle\Tests\Services\ProductVariant\Version1\Update;

use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\ClientException;
use JMS\Serializer\SerializerInterface;
use Lifestyle\DataCollector\DataCollector;
use Lifestyle\Sylius\RestApiClientBundle\Services\ProductVariant\Version1\Update\Handler;
use Lifestyle\Sylius\RestApiClientBundle\Services\ProductVariant\Version1\Update\Mapper;
use Lifestyle\Sylius\RestApiClientBundle\Exception\RestRequestFailedException;
use Lifestyle\Sylius\RestApiClientBundle\Services\ProductVariant\Version1\Update\Model\Request\Product;
use Psr\Http\Message\RequestInterface;
use Psr\Log\LoggerInterface;
use PHPUnit\Framework\TestCase;
use GuzzleHttp\Client;
use GuzzleHttp\Handler\MockHandler;
use GuzzleHttp\HandlerStack;
use GuzzleHttp\Psr7\Response;
use Symfony\Component\EventDispatcher\EventDispatcher;

/**
 * Class HandlerTest
 * @package Lifestyle\Sylius\RestApiClientBundle\Tests\Services\ProductVariant\Version1\Update
 */
class HandlerTest extends TestCase
{
    public function testUpdateSuccess()
    {
        $serializer = $this->getMockBuilder(SerializerInterface::class)->getMock();
        $logger = $this->getMockBuilder(LoggerInterface::class)->getMock();
        $mapper = $this->getMockBuilder(Mapper::class)->getMock();
        $productVariantModel = $this->getMockBuilder(Product::class)->getMock();
        $eventDispatcher = new EventDispatcher();

        $mapper->expects($this->once())
            ->method('mapRequestModel')
            ->willReturn($productVariantModel);

        // Create a mock and queue the responses.
        $mock = new MockHandler([
            new Response(204, []),
        ]);

        $handler = HandlerStack::create($mock);
        $client = new Client(['handler' => $handler]);

        $collector = new DataCollector();
        $collector->addItem('parentId', '123456');
        $collector->addItem('id', '123456');

        $updateHandler = new Handler($serializer, $logger, $client, $mapper, $eventDispatcher);
        $this->assertTrue($updateHandler->update($collector));
    }

    public function testUpdateRequestUnexpectedClientException()
    {
        $this->expectException(RestRequestFailedException::class);
        $this->expectExceptionMessage('Could not update product variant object!');

        $serializer = $this->getMockBuilder(SerializerInterface::class)->getMock();
        $logger = $this->getMockBuilder(LoggerInterface::class)->getMock();
        $mapper = $this->getMockBuilder(Mapper::class)->getMock();
        $productVariantModel = $this->getMockBuilder(Product::class)->getMock();
        $eventDispatcher = new EventDispatcher();

        $mapper->expects($this->once())
            ->method('mapRequestModel')
            ->willReturn($productVariantModel);

        // Create a mock and queue the responses.
        $mock = new MockHandler([
            new ClientException("Mocked Client Exception", $this->getMockBuilder(RequestInterface::class)->getMock(), new Response(999)),
        ]);

        $handler = HandlerStack::create($mock);
        $client = new Client(['handler' => $handler]);

        $collector = new DataCollector();
        $collector->addItem('parentId', '123456');
        $collector->addItem('id', '123456');

        $updateHandler = new Handler($serializer, $logger, $client, $mapper, $eventDispatcher);
        $updateHandler->update($collector);
    }

    public function testUpdateRequestUnexpectedException()
    {
        $this->expectException(RestRequestFailedException::class);
        $this->expectExceptionMessage('Could not update product variant object! - unexpected mocked exception');

        $serializer = $this->getMockBuilder(SerializerInterface::class)->getMock();
        $logger = $this->getMockBuilder(LoggerInterface::class)->getMock();
        $mapper = $this->getMockBuilder(Mapper::class)->getMock();
        $productVariantModel = $this->getMockBuilder(Product::class)->getMock();
        $eventDispatcher = new EventDispatcher();

        $mapper->expects($this->once())
            ->method('mapRequestModel')
            ->willReturn($productVariantModel);

        // Create a mock and queue the responses.
        $mock = new MockHandler([
            new \Exception("unexpected mocked exception"),
        ]);

        $handler = HandlerStack::create($mock);
        $client = new Client(['handler' => $handler]);

        $collector = new DataCollector();
        $collector->addItem('parentId', '123456');
        $collector->addItem('id', '123456');

        $updateHandler = new Handler($serializer, $logger, $client, $mapper, $eventDispatcher);
        $updateHandler->update($collector);
    }

    public function testUpdateNoParentCodeException()
    {
        $this->expectException(RestRequestFailedException::class);
        $this->expectExceptionMessage('An unexpected error occurred while updating the object: Product or variant code is missing.');

        $serializer = $this->getMockBuilder(SerializerInterface::class)->getMock();
        $logger = $this->getMockBuilder(LoggerInterface::class)->getMock();
        $mapper = $this->getMockBuilder(Mapper::class)->getMock();
        $eventDispatcher = new EventDispatcher();

        // Create a mock and queue the responses.
        $mock = new MockHandler([]);

        $handler = HandlerStack::create($mock);
        $client = new Client(['handler' => $handler]);

        $collector = new DataCollector();
        $collector->addItem('id', '123456');

        $updateHandler = new Handler($serializer, $logger, $client, $mapper, $eventDispatcher);
        $updateHandler->update($collector);
    }

    public function testUpdateNoVariantCodeException()
    {
        $this->expectException(RestRequestFailedException::class);
        $this->expectExceptionMessage('An unexpected error occurred while updating the object: Product or variant code is missing.');

        $serializer = $this->getMockBuilder(SerializerInterface::class)->getMock();
        $logger = $this->getMockBuilder(LoggerInterface::class)->getMock();
        $mapper = $this->getMockBuilder(Mapper::class)->getMock();
        $eventDispatcher = new EventDispatcher();

        // Create a mock and queue the responses.
        $mock = new MockHandler([]);

        $handler = HandlerStack::create($mock);
        $client = new Client(['handler' => $handler]);

        $collector = new DataCollector();
        $collector->addItem('parentId', '123456');

        $updateHandler = new Handler($serializer, $logger, $client, $mapper, $eventDispatcher);
        $updateHandler->update($collector);
    }

    public function testBuildUri()
    {
        $serializer = $this->getMockBuilder(SerializerInterface::class)->getMock();
        $logger = $this->getMockBuilder(LoggerInterface::class)->getMock();
        $client = $this->getMockBuilder(ClientInterface::class)->getMock();
        $mapper = $this->getMockBuilder(Mapper::class)->getMock();
        $eventDispatcher = new EventDispatcher();

        $updateHandler = new Handler($serializer, $logger, $client, $mapper, $eventDispatcher);

        $class = new \ReflectionClass(Handler::class);
        $method = $class->getMethod('buildUri');
        $method->setAccessible(true);

        $uri = $method->invoke($updateHandler, 'productCode1234', 'variantCode1234');
        $this->assertSame('/api/v1/products/productCode1234/variants/variantCode1234', $uri);
    }
}
