<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2018 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Lifestyle\Sylius\RestApiClientBundle\Services\ProductVariant\Version1\GetCollection;

use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\BadResponseException;
use GuzzleHttp\Exception\GuzzleException;
use JMS\Serializer\SerializerInterface;
use JMS\Serializer\Exception\Exception as SerializerException;
use Lifestyle\Sylius\RestApiClientBundle\Exception\RestRequestFailedException;
use Lifestyle\Sylius\RestApiClientBundle\Services\ProductVariant\Version1\GetCollection\Model\Response\Response;
use Psr\Http\Message\ResponseInterface;
use Psr\Log\LoggerInterface;

/**
 * Class Handler
 *
 * @copyright  2018 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package Lifestyle\Sylius\RestApiClientBundle\Services\ProductVariant\Version1\GetCollection
 */
class Handler
{
    const API_CGET_URI = '/api/v1/products/%s/variants';

    /**
     * @var SerializerInterface
     */
    private $serializer;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * @var ClientInterface
     */
    private $client;

    /**
     * Handler constructor.
     * @param SerializerInterface $serializer
     * @param LoggerInterface $logger
     * @param ClientInterface $client
     */
    public function __construct(SerializerInterface $serializer, LoggerInterface $logger, ClientInterface $client)
    {
        $this->serializer = $serializer;
        $this->logger = $logger;
        $this->client = $client;
    }

    /**
     * @param string $productCode
     * @param int $limit Default: 10
     * @return Response|false
     * @throws RestRequestFailedException
     */
    public function getAll(string $productCode, int $limit = 10)
    {
        if (0 === strlen($productCode)) {
            throw new RestRequestFailedException(
                'An unexpected error occurred while fetching the object: product-code empty.'
            );
        }

        $uri = $this->buildUri($productCode);
        $options = [
            'query' => ['limit' => $limit]
        ];

        $this->logger->info(sprintf('Send GET request %s', $uri));

        try {
            $response = $this->client->request('GET', $uri, $options);
        } catch (BadResponseException $exception) {
            if ($exception->hasResponse() && 404 === $exception->getResponse()->getStatusCode()) {
                $this->logger->warning(sprintf('Product (id: %s) does not exist in Sylius.', $productCode));
                return false;
            }
            throw new RestRequestFailedException(
                sprintf('An unexpected error occurred while fetching the object: %s', $exception->getMessage()),
                $exception->getCode(),
                $exception
            );
        } catch (GuzzleException $exception) {
            throw new RestRequestFailedException(
                sprintf('An unexpected error occurred while fetching the object: %s', $exception->getMessage()),
                $exception->getCode(),
                $exception
            );
        }

        return 200 === $response->getStatusCode() ? $this->getResponseModel($response) : false;
    }

    /**
     * @param string $productCode
     * @return string
     */
    private function buildUri(string $productCode)
    {
        return sprintf(self::API_CGET_URI, $productCode);
    }

    /**
     * @param ResponseInterface $response
     * @return Response | null
     * @throws RestRequestFailedException
     */
    private function getResponseModel(ResponseInterface $response)
    {
        try {
            $response = $this->serializer->deserialize($response->getBody()->getContents(), Response::class, 'json');
        } catch (SerializerException $exception) {
            throw new RestRequestFailedException(
                'An unexpected error occurred while parsing response from Sylius',
                max(400, $response->getStatusCode()),
                $exception
            );
        }

        return $response instanceof Response ? $response : null;
    }
}
