<?php

/**
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2017 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Tests\Lifestyle\Sylius\RestApiClientBundle\Services\ProductAttributes\Version1;

use Lifestyle\DataCollector\DataCollector;
use Lifestyle\Sylius\RestApiClientBundle\Services\ProductAttributes\Version1\Create\Handler as CreateHandler;
use Lifestyle\Sylius\RestApiClientBundle\Services\ProductAttributes\Version1\Delete\Handler as DeleteHandler;
use Lifestyle\Sylius\RestApiClientBundle\Services\ProductAttributes\Version1\GeneralModels\Response\ProductAttributes;
use Lifestyle\Sylius\RestApiClientBundle\Services\ProductAttributes\Version1\GeneralModels\Value;
use Lifestyle\Sylius\RestApiClientBundle\Services\ProductAttributes\Version1\Get\Handler as GetHandler;
use Lifestyle\Sylius\RestApiClientBundle\Services\ProductAttributes\Version1\Handler;
use Lifestyle\Sylius\RestApiClientBundle\Services\ProductAttributes\Version1\Update\Handler as UpdateHandler;
use Lifestyle\Sylius\RestApiClientBundle\Services\ProductAttributes\Version1\Factory;
use PHPUnit\Framework\TestCase;
use Psr\Log\LoggerInterface;

/**
 * Class HandlerTest
 * @package Tests\Lifestyle\Sylius\RestApiClientBundle\Services\ProductAttributes\Version1
 */
class HandlerTest extends TestCase
{
    /**
     * @var Handler
     */
    private $productHandler;

    /**
     * @var CreateHandler|\PHPUnit_Framework_MockObject_MockObject
     */
    private $createHandler;

    /**
     * @var DeleteHandler|\PHPUnit_Framework_MockObject_MockObject
     */
    private $deleteHandler;

    /**
     * @var GetHandler|\PHPUnit_Framework_MockObject_MockObject
     */
    private $getHandler;

    /**
     * @var UpdateHandler|\PHPUnit_Framework_MockObject_MockObject
     */
    private $updateHandler;

    protected function setUp()
    {
        parent::setUp();

        $this->createHandler = $this->getMockBuilder(CreateHandler::class)
            ->disableOriginalConstructor()
            ->getMock();
        $this->deleteHandler = $this->getMockBuilder(DeleteHandler::class)
            ->disableOriginalConstructor()
            ->setMethods(['deleteByCode'])
            ->getMock();
        $this->getHandler = $this->getMockBuilder(GetHandler::class)
            ->disableOriginalConstructor()
            ->setMethods(['exists', 'getByCode'])
            ->getMock();
        $this->updateHandler = $this->getMockBuilder(UpdateHandler::class)
            ->disableOriginalConstructor()
            ->setMethods(['update'])
            ->getMock();

        $serviceFactory = new Factory($this->createHandler, $this->deleteHandler, $this->getHandler, $this->updateHandler);
        $logger = $this->getMockBuilder(LoggerInterface::class)->getMock();

        $this->productHandler = new Handler($serviceFactory, $logger);
    }

    public function testIsResponsible()
    {
        $dataCollector = new DataCollector();
        $dataCollector->addItem('className', 'Option');

        // @TODO Why should an attribute handler listen to option-events?
        $this->assertTrue($this->productHandler->isResponsible($dataCollector));
    }

    public function testIsNotResponsible()
    {
        $dataCollector = new DataCollector();
        $dataCollector->addItem('className', 'SomeOtherClass');

        $this->assertFalse($this->productHandler->isResponsible($dataCollector));
    }

    public function testCreate()
    {
        $this->getHandler->expects($this->once())
            ->method('getByCode')
            ->willReturn(false);

        $this->createHandler->expects($this->once())
            ->method('create');

        $dataCollector = new DataCollector();

        $this->productHandler->createOrUpdate($dataCollector);
    }

    public function testCreateOrUpdate()
    {
        $responseModel = new ProductAttributes();
        $value = new Value();
        $value->setCode('size_S');
        $responseModel->setValues([$value]);

        $this->getHandler->expects($this->once())
            ->method('getByCode')
            ->willReturn($responseModel);

        $dataCollector = new DataCollector();
        $dataCollector->addItem('optionCode', 'size');
        $dataCollector->addItem('optionValue', 'S');
        $this->productHandler->createOrUpdate($dataCollector);
        // Make sure the corresponding version is added.
        $this->assertSame('size', $dataCollector->getItemValue('optionCode'));
        $this->assertSame('S', $dataCollector->getItemValue('optionValue'));
    }

    public function testCreateOrUpdateValueExists()
    {
        $responseModel = new ProductAttributes();

        $this->getHandler->expects($this->once())
            ->method('getByCode')
            ->willReturn($responseModel);

        $dataCollector = new DataCollector();
        $dataCollector->addItem('optionCode', 'size');
        $dataCollector->addItem('optionValue', 'S');
        $this->productHandler->createOrUpdate($dataCollector);
        // Make sure the corresponding version is added.
        $this->assertSame('size', $dataCollector->getItemValue('optionCode'));
        $this->assertSame('S', $dataCollector->getItemValue('optionValue'));
    }

    public function testCreateOrUpdateOptions()
    {
        $dataCollector = new DataCollector();
        $dataCollector->addItem('options', ['size' => 'S']);
        $this->productHandler->createOrUpdateOptions($dataCollector);
        // Make sure the corresponding version is added.
        $this->assertSame(['size' => 'S'], $dataCollector->getItemValue('options'));
    }

    public function testDelete()
    {
        $optionCode = 'size';

        $this->getHandler->expects($this->once())
            ->method('exists')
            ->with($optionCode)
            ->willReturn(true);

        $this->deleteHandler->expects($this->once())
            ->method('deleteByCode')
            ->with($optionCode);

        $dataCollector = new DataCollector();
        $dataCollector->addItem('code', $optionCode);

        $this->productHandler->delete($dataCollector);
    }

    public function testDoNotDelete()
    {
        $optionCode = 'size';

        $this->getHandler->expects($this->once())
            ->method('exists')
            ->with($optionCode)
            ->willReturn(false);

        $this->deleteHandler->expects($this->never())
            ->method('deleteByCode');

        $dataCollector = new DataCollector();
        $dataCollector->addItem('code', $optionCode);

        $this->productHandler->delete($dataCollector);
    }
}
