<?php

/**
 * Class Mapper
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2017 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Lifestyle\Sylius\RestApiClientBundle\Services\Product\Version1\Update;

use Lifestyle\DataCollector\DataCollectorInterface;
use Lifestyle\Sylius\RestApiClientBundle\Services\Product\Version1\Update\Model\Request\Association;
use Lifestyle\Sylius\RestApiClientBundle\Services\Product\Version1\Update\Model\Request\Product;
use Lifestyle\Sylius\RestApiClientBundle\Services\Product\Version1\Update\Model\Request\Translation;
use Lifestyle\Sylius\RestApiClientBundle\Services\Product\Version1\Update\Model\Request\Attribute;
use Pimcore\Model\DataObject;


/**
 * Class Mapper
 * @package Lifestyle\Sylius\RestApiClientBundle\Services\Product\Version1\Update
 */
class Mapper
{
    const ASSOCIATION_TYPE_SIMILAR_PRODUCTS = 'similar_products';

    /**
     * @param DataCollectorInterface $collector
     * @return Product
     */
    public function mapRequestModel(DataCollectorInterface $collector)
    {
        $product = new Product();
        $product->setCode($collector->getItemValue('id'));
        $product->setMainTaxon($collector->getItemValue('mainTaxon'));

        $productTaxon = $collector->getItemValue('productTaxon');
        if (is_array($productTaxon)) {
            $productTaxon = join(',', $productTaxon);
        }
        $product->setProductTaxon($productTaxon);

        $channels = $collector->getItemValue('channels');
        if (is_array($channels)) {
            $channels = array_values($channels);
        }
        $product->setChannels($channels);

        if ($collector->hasItem('enabled')) {
            $product->setEnabled((bool)$collector->getItemValue('enabled'));
        }

        $product = $this->addSimilarProducts($collector, $product);
        $product = $this->mapLocalizedFields($collector, $product);
        $product = $this->mapAttributes($collector, $product);

        // create product options if not exist
        if ($collector->hasItem('options') && is_array($collector->getItemValue('options'))) {
            $product->setOptions(array_values($collector->getItemValue('options')));
        }

        return $product;
    }

    /**
     * @param DataCollectorInterface $collector
     * @param Product $product
     * @return Product
     */
    private function addSimilarProducts(DataCollectorInterface $collector, Product $product): Product
    {
        if (!$collector->hasItem('similarProducts')) {
            return $product;
        }

        $similarProducts = $collector->getItemValue('similarProducts');
        $associationValue = is_array($similarProducts) ? implode(',', $similarProducts) : $similarProducts;

        $association = new Association();
        $association
            ->setType(self::ASSOCIATION_TYPE_SIMILAR_PRODUCTS)
            ->setValue($associationValue);

        $product->addAssociation($association);

        return $product;
    }

    /**
     * @param DataCollectorInterface $collector
     * @param Product $product
     * @return Product
     */
    private function mapLocalizedFields(DataCollectorInterface $collector, Product $product): Product
    {
        $locales = $collector->getCollectedLocales();
        foreach ($locales as $locale) {
            $translation = new Translation();

            $this->mapTranslationField($translation, $collector, 'name', $locale);
            $this->mapTranslationField($translation, $collector, 'slug', $locale);
            $this->mapTranslationField($translation, $collector, 'description', $locale);
            $this->mapTranslationField($translation, $collector, 'shortDescription', $locale);
            $this->mapTranslationField($translation, $collector, 'metaKeywords', $locale);
            $this->mapTranslationField($translation, $collector, 'metaDescription', $locale);

            $translation->setLocale($locale);
            $product->addTranslation($translation);
        }

        return $product;
    }

    /**
     * @param Translation $translation
     * @param DataCollectorInterface $collector
     * @param string $fieldName
     * @param string $lang
     */
    private function mapTranslationField(Translation $translation, DataCollectorInterface $collector, $fieldName, $lang)
    {
        if ($collector->hasItem($fieldName, $lang)) {
            $method = 'set' . ucfirst(strtolower($fieldName));
            $translation->$method($collector->getItemValue($fieldName, $lang));
        }
    }

    /**
     * @param DataCollectorInterface $collector
     * @param Product $product
     * @return Product
     */
    private function mapAttributes(DataCollectorInterface $collector, Product $product): Product
    {
        if (!$collector->hasItem('attributes')) {
            return $product;
        }

        $locales = $collector->getCollectedLocales();
        $attributes = $collector->getItemValue('attributes');
        foreach ((is_array($attributes) ? $attributes : [$attributes]) as $attributeCode) {
            foreach ($locales as $locale) {
                if (!$collector->hasItem('attributes_translated_' . $attributeCode, $locale)) {
                    continue;
                }
                $attributeValue = $collector->getItemValue('attributes_translated_' . $attributeCode, $locale);
                $attribute = new Attribute();
                $attribute->setAttribute($attributeCode);
                $attribute->setLocaleCode($locale);
                $attribute->setValue(is_array($attributeValue) ? join(', ', $attributeValue) : $attributeValue);
                $product->addAttribute($attribute);
            }
        }

        return $product;
    }
}
