<?php

/**
 * Class Handler
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2017 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Lifestyle\Sylius\RestApiClientBundle\Services\Taxon\Version1\Get;

use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\ClientException;
use JMS\Serializer\SerializerInterface;
use Lifestyle\Sylius\RestApiClientBundle\Exception\RestRequestFailedException;
use Psr\Log\LoggerInterface;

/**
 * Class Handler
 * @package Lifestyle\Sylius\RestApiClientBundle\Services\Taxon\Version1\Get
 */
class Handler
{
    const API_GET_URI = '/api/v1/taxons/%s';

    /**
     * @var SerializerInterface
     */
    private $serializer;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * @var ClientInterface
     */
    private $client;

    /**
     * @param SerializerInterface $serializer
     * @param LoggerInterface $logger
     * @param ClientInterface $client
     */
    public function __construct(SerializerInterface $serializer, LoggerInterface $logger, ClientInterface $client)
    {
        $this->serializer = $serializer;
        $this->logger = $logger;
        $this->client = $client;
    }

    /**
     * @param $code
     * @return bool
     * @throws RestRequestFailedException
     * @throws \GuzzleHttp\Exception\GuzzleException
     */
    public function exists($code)
    {
        try {
            $response = $this->getByCode($code);
        } catch (ClientException $exception) {
            if (404 === $exception->getResponse()->getStatusCode()) {
                return false;
            }
            throw new RestRequestFailedException(
                sprintf('An unexpected error occurred while saving the object: %s', $exception->getMessage()),
                $exception->getCode()
            );
        } catch (\Exception $exception) {
            throw new RestRequestFailedException(
                sprintf('An unexpected error occurred while saving the object: %s', $exception->getMessage()),
                $exception->getCode()
            );
        }
        return 200 === $response->getStatusCode();
    }

    /**
     * @param $code
     * @return \Psr\Http\Message\ResponseInterface
     * @throws RestRequestFailedException
     * @throws \GuzzleHttp\Exception\GuzzleException
     */
    private function getByCode($code)
    {
        if (!strlen($code)) {
            throw new RestRequestFailedException(
                'An unexpected error occurred while saving the object: Taxon code is missing.'
            );
        }

        $uri = $this->buildUri($code);

        $this->logger->info(sprintf('Send GET request %s', $uri));

        return $this->client->request('GET', $uri);
    }

    /**
     * @param string $code
     * @return string
     */
    private function buildUri($code)
    {
        return sprintf(self::API_GET_URI, $code);
    }
}
