<?php

/**
 * Class HandlerTest
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2017 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Lifestyle\Sylius\RestApiClientBundle\Tests\Services\Taxon\Version1\Get;

use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\ClientException;
use JMS\Serializer\SerializerInterface;
use Lifestyle\Sylius\RestApiClientBundle\Services\Taxon\Version1\Get\Handler;
use Lifestyle\Sylius\RestApiClientBundle\Exception\RestRequestFailedException;
use Psr\Http\Message\RequestInterface;
use Psr\Log\LoggerInterface;
use PHPUnit\Framework\TestCase;
use GuzzleHttp\Client;
use GuzzleHttp\Handler\MockHandler;
use GuzzleHttp\HandlerStack;
use GuzzleHttp\Psr7\Response;

/**
 * Class HandlerTest
 * @package Lifestyle\Sylius\RestApiClientBundle\Tests\Services\Taxon\Version1\Get
 */
class HandlerTest extends TestCase
{
    public function testExists()
    {
        $serializer = $this->getMockBuilder(SerializerInterface::class)->getMock();
        $logger = $this->getMockBuilder(LoggerInterface::class)->getMock();

        // Create a mock and queue the responses.
        $mock = new MockHandler([
            new Response(200, []),
        ]);

        $handler = HandlerStack::create($mock);
        $client = new Client(['handler' => $handler]);

        $getHandler = new Handler($serializer, $logger, $client);
        $this->assertTrue($getHandler->exists('unitTest123'));
    }

    public function testDoesNotExists()
    {
        $serializer = $this->getMockBuilder(SerializerInterface::class)->getMock();
        $logger = $this->getMockBuilder(LoggerInterface::class)->getMock();

        // Create a mock and queue the responses.
        $mock = new MockHandler([
            new ClientException("", $this->getMockBuilder(RequestInterface::class)->getMock(), new Response(404)),
        ]);

        $handler = HandlerStack::create($mock);
        $client = new Client(['handler' => $handler]);

        $getHandler = new Handler($serializer, $logger, $client);
        $this->assertFalse($getHandler->exists('unitTest123'));
    }

    public function testGetRequestUnexpectedClientException()
    {
        $this->expectException(RestRequestFailedException::class);

        $serializer = $this->getMockBuilder(SerializerInterface::class)->getMock();
        $logger = $this->getMockBuilder(LoggerInterface::class)->getMock();

        // Create a mock and queue the responses.
        $mock = new MockHandler([
            new ClientException("", $this->getMockBuilder(RequestInterface::class)->getMock(), new Response(999)),
        ]);

        $handler = HandlerStack::create($mock);
        $client = new Client(['handler' => $handler]);

        $getHandler = new Handler($serializer, $logger, $client);
        $getHandler->exists('unitTest123');
    }

    public function testGetRequestUnexpectedException()
    {
        $this->expectException(RestRequestFailedException::class);

        $serializer = $this->getMockBuilder(SerializerInterface::class)->getMock();
        $logger = $this->getMockBuilder(LoggerInterface::class)->getMock();

        // Create a mock and queue the responses.
        $mock = new MockHandler([
            new \Exception("unexpected mocked exception"),
        ]);

        $handler = HandlerStack::create($mock);
        $client = new Client(['handler' => $handler]);

        $getHandler = new Handler($serializer, $logger, $client);
        $getHandler->exists('unitTest123');
    }

    public function testGetByCodeThrowsNoCodeException()
    {
        $this->expectException(RestRequestFailedException::class);

        $serializer = $this->getMockBuilder(SerializerInterface::class)->getMock();
        $logger = $this->getMockBuilder(LoggerInterface::class)->getMock();

        // Create a mock and queue the responses.
        $mock = new MockHandler([]);

        $handler = HandlerStack::create($mock);
        $client = new Client(['handler' => $handler]);

        $getHandler = new Handler($serializer, $logger, $client);
        $getHandler->exists(null);
    }


    public function testBuildUri()
    {
        $serializer = $this->getMockBuilder(SerializerInterface::class)->getMock();
        $logger = $this->getMockBuilder(LoggerInterface::class)->getMock();
        $client = $this->getMockBuilder(ClientInterface::class)->getMock();

        $updateHandler = new Handler($serializer, $logger, $client);

        $class = new \ReflectionClass(Handler::class);
        $method = $class->getMethod('buildUri');
        $method->setAccessible(true);

        $uri = $method->invoke($updateHandler, 'taxonCode12345');
        $this->assertSame('/api/v1/taxons/taxonCode12345', $uri);
    }
}
