<?php

/**
 * Class HandlerTest
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2017 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Lifestyle\Sylius\RestApiClientBundle\Tests\Services\Taxon\Version1;

use Lifestyle\DataCollector\DataCollector;
use Lifestyle\Sylius\RestApiClientBundle\Services\Taxon\Version1\Create\Handler as CreateHandler;
use Lifestyle\Sylius\RestApiClientBundle\Services\Taxon\Version1\Delete\Handler as DeleteHandler;
use Lifestyle\Sylius\RestApiClientBundle\Services\Taxon\Version1\Get\Handler as GetHandler;
use Lifestyle\Sylius\RestApiClientBundle\Services\Taxon\Version1\Handler;
use Lifestyle\Sylius\RestApiClientBundle\Services\Taxon\Version1\Update\Handler as UpdateHandler;
use Lifestyle\Sylius\RestApiClientBundle\Services\Taxon\Version1\Factory;
use PHPUnit\Framework\TestCase;
use Psr\Log\LoggerInterface;

/**
 * Class HandlerTest
 * @package Lifestyle\Sylius\RestApiClientBundle\Tests\Services\Taxon\Version1
 */
class HandlerTest extends TestCase
{
    /**
     * @var Handler
     */
    private $taxonHandler;

    /**
     * @var CreateHandler|\PHPUnit_Framework_MockObject_MockObject
     */
    private $createHandler;

    /**
     * @var DeleteHandler|\PHPUnit_Framework_MockObject_MockObject
     */
    private $deleteHandler;

    /**
     * @var GetHandler|\PHPUnit_Framework_MockObject_MockObject
     */
    private $getHandler;

    /**
     * @var UpdateHandler|\PHPUnit_Framework_MockObject_MockObject
     */
    private $updateHandler;

    protected function setUp()
    {
        parent::setUp();

        $this->createHandler = $this->getMockBuilder(CreateHandler::class)
            ->disableOriginalConstructor()
            ->getMock();
        $this->deleteHandler = $this->getMockBuilder(DeleteHandler::class)
            ->disableOriginalConstructor()
            ->setMethods(['deleteByCode'])
            ->getMock();
        $this->getHandler = $this->getMockBuilder(GetHandler::class)
            ->disableOriginalConstructor()
            ->setMethods(['exists'])
            ->getMock();
        $this->updateHandler = $this->getMockBuilder(UpdateHandler::class)
            ->disableOriginalConstructor()
            ->setMethods(['update'])
            ->getMock();

        $serviceFactory = new Factory($this->createHandler, $this->deleteHandler, $this->getHandler, $this->updateHandler);
        $logger = $this->getMockBuilder(LoggerInterface::class)->getMock();

        $this->taxonHandler = new Handler($serviceFactory, $logger);
    }

    public function testIsResponsible()
    {
        $dataCollector = new DataCollector();
        $dataCollector->addItem('className', 'Taxon');

        $this->assertTrue($this->taxonHandler->isResponsible($dataCollector));
    }

    public function testIsNotResponsible()
    {
        $dataCollector = new DataCollector();
        $dataCollector->addItem('className', 'SomeOtherClass');

        $this->assertFalse($this->taxonHandler->isResponsible($dataCollector));
    }

    public function testCreate()
    {
        $this->getHandler->expects($this->once())
            ->method('exists')
            ->willReturn(false);

        $this->createHandler->expects($this->once())
            ->method('create');

        $dataCollector = new DataCollector();

        $this->taxonHandler->createOrUpdate($dataCollector);
    }

    public function testUpdate()
    {
        $this->getHandler->expects($this->once())
            ->method('exists')
            ->willReturn(true);

        $this->updateHandler->expects($this->once())
            ->method('update');

        $dataCollector = new DataCollector();

        $this->taxonHandler->createOrUpdate($dataCollector);
    }

    public function testDelete()
    {
        $taxonCode = 'unitTest123';

        $this->getHandler->expects($this->once())
            ->method('exists')
            ->with($taxonCode)
            ->willReturn(true);

        $this->deleteHandler->expects($this->once())
            ->method('deleteByCode')
            ->with($taxonCode);

        $dataCollector = new DataCollector();
        $dataCollector->addItem('id', $taxonCode);

        $this->taxonHandler->delete($dataCollector);
    }

    public function testDoNotDelete()
    {
        $taxonCode = 'unitTest123';

        $this->getHandler->expects($this->once())
            ->method('exists')
            ->with($taxonCode)
            ->willReturn(false);

        $this->deleteHandler->expects($this->never())
            ->method('deleteByCode');

        $dataCollector = new DataCollector();
        $dataCollector->addItem('id', $taxonCode);

        $this->taxonHandler->delete($dataCollector);
    }
}
