<?php

/**
 * Class ConfigTest
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2017 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Lifestyle\Sylius\RestApiClientBundle\Tests\Services;

use Lifestyle\Sylius\RestApiClientBundle\Exception\InvalidConfigurationException;
use Lifestyle\Sylius\RestApiClientBundle\Exception\RestRequestFailedException;
use Lifestyle\Sylius\RestApiClientBundle\Services\ConfigValidator;
use PHPUnit\Framework\TestCase;

/**
 * Class ConfigTest
 * @package Lifestyle\Sylius\RestApiClientBundle\Tests\Services
 */
class ConfigTest extends TestCase
{
    const EXPECTED_EXCEPTION_MESSAGE = 'Invalid or missing plugin configuration for sylius export handler!';

    public function testConfigIsInvalidNoBaseUri()
    {
        $this->expectException(InvalidConfigurationException::class);
        $this->expectExceptionMessage(self::EXPECTED_EXCEPTION_MESSAGE);

        $config = new ConfigValidator(
            null,
            'unit_test_user',
            'unit_test_password',
            '/unit/test/oauth',
            'unit_test_client_id',
            'unit_test_client_secret'
        );

        $config->validate();
    }

    public function testConfigIsInvalidNoUserName()
    {
        $this->expectException(InvalidConfigurationException::class);
        $this->expectExceptionMessage(self::EXPECTED_EXCEPTION_MESSAGE);

        $config = new ConfigValidator(
            'htttttp:///unit/test/base',
            '',
            'unit_test_password',
            '/unit/test/oauth',
            'unit_test_client_id',
            'unit_test_client_secret'
        );

        $config->validate();
    }

    public function testConfigIsInvalidNoPassword()
    {
        $this->expectException(InvalidConfigurationException::class);
        $this->expectExceptionMessage(self::EXPECTED_EXCEPTION_MESSAGE);

        $config = new ConfigValidator(
            'htttttp:///unit/test/base',
            'unit_test_user',
            '',
            '/unit/test/oauth',
            'unit_test_client_id',
            'unit_test_client_secret'
        );

        $config->validate();
    }

    public function testConfigIsInvalidNoTokenUrl()
    {
        $this->expectException(InvalidConfigurationException::class);
        $this->expectExceptionMessage(self::EXPECTED_EXCEPTION_MESSAGE);

        $config = new ConfigValidator(
            'htttttp:///unit/test/base',
            'unit_test_user',
            'unit_test_password',
            null,
            'unit_test_client_id',
            'unit_test_client_secret'
        );

        $config->validate();
    }

    public function testConfigIsInvalidNoClientId()
    {
        $this->expectException(InvalidConfigurationException::class);
        $this->expectExceptionMessage(self::EXPECTED_EXCEPTION_MESSAGE);

        $config = new ConfigValidator(
            'htttttp:///unit/test/base',
            'unit_test_user',
            'unit_test_password',
            '/unit/test/oauth',
            '',
            'unit_test_client_secret'
        );

        $config->validate();
    }

    public function testConfigIsInvalidNoClientSecret()
    {
        $this->expectException(InvalidConfigurationException::class);
        $this->expectExceptionMessage(self::EXPECTED_EXCEPTION_MESSAGE);

        $config = new ConfigValidator(
            'htttttp:///unit/test/base',
            'unit_test_user',
            'unit_test_password',
            '/unit/test/oauth',
            'unit_test_client_id',
            null
        );

        $config->validate();
    }

    public function testConfigIsValid()
    {
        $config = new ConfigValidator(
            'htttttp:///unit/test/base',
            'unit_test_user',
            'unit_test_password',
            '/unit/test/oauth',
            'unit_test_client_id',
            'unit_test_client_secret'
        );

        try {
            $config->validate();
        } catch (\Exception $exception) {
            $this->fail('Config is valid but an exception was thrown!');
        }

        $this->assertTrue(true);
    }

    public function testConfigToArray()
    {
        $arrayConfig = [
            'apiBaseUri' => 'htttttp:///unit/test/base',
            'apiUserName' => 'unit_test_user',
            'apiPassword' => 'unit_test_password',
            'apiTokenUrl' => '/unit/test/oauth',
            'apiClientId' => 'unit_test_client_id',
            'apiClientSecret' => 'unit_test_client_secret',
        ];

        $config = new ConfigValidator(
            $arrayConfig['apiBaseUri'],
            $arrayConfig['apiUserName'],
            $arrayConfig['apiPassword'],
            $arrayConfig['apiTokenUrl'],
            $arrayConfig['apiClientId'],
            $arrayConfig['apiClientSecret']
        );

        // Just to be sure
        $config->validate();

        $this->assertSame($arrayConfig, $config->toArray());
    }
}
