<?php

/**
 * Class Handler
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2017 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Lifestyle\Sylius\RestApiClientBundle\Services\Taxon\Version1\Update;

use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\ClientException;
use JMS\Serializer\SerializerInterface;
use Lifestyle\DataCollector\DataCollectorInterface;
use Lifestyle\Sylius\RestApiClientBundle\Exception\RestRequestFailedException;
use Lifestyle\Sylius\RestApiClientBundle\Helper\Mapper\RequestModelValidatorTrait;
use Psr\Log\LoggerInterface;
use Symfony\Component\Validator\Validator\ValidatorInterface;

/**
 * Class Handler
 * @package Lifestyle\Sylius\RestApiClientBundle\Services\Taxon\Version1\Update
 */
class Handler
{
    use RequestModelValidatorTrait;

    const API_UPDATE_URI = '/api/v1/taxons/%s';
    const CLIENT_EXCEPTION_MESSAGE = 'Could not update taxon object!';

    /**
     * @var SerializerInterface
     */
    private $serializer;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * @var ClientInterface
     */
    private $client;

    /**
     * @var Mapper
     */
    private $mapper;

    /**
     * @var ValidatorInterface
     */
    private $validator;

    /**
     * Handler constructor.
     * @param SerializerInterface $serializer
     * @param LoggerInterface $logger
     * @param ClientInterface $client
     * @param Mapper $mapper
     * @param ValidatorInterface $validator
     */
    public function __construct(
        SerializerInterface $serializer,
        LoggerInterface $logger,
        ClientInterface $client,
        Mapper $mapper,
        ValidatorInterface $validator
    ) {
        $this->serializer = $serializer;
        $this->logger = $logger;
        $this->client = $client;
        $this->mapper = $mapper;
        $this->validator = $validator;
    }

    /**
     * @param DataCollectorInterface $collector
     * @return bool
     * @throws RestRequestFailedException
     * @throws \GuzzleHttp\Exception\GuzzleException
     */
    public function update(DataCollectorInterface $collector)
    {
        if (!strlen($collector->getItemValue('id'))) {
            throw new RestRequestFailedException(
                'An unexpected error occurred while updating the object: Taxon code is missing.'
            );
        }

        $taxon = $this->mapper->mapRequestModel($collector);
        $this->validateRequest($this->validator, self::CLIENT_EXCEPTION_MESSAGE, $taxon);

        $options = [
            'body' => $this->serializer->serialize($taxon, 'json'),
            'headers' => ['Content-type' => 'application/json']
        ];

        $uri = $this->buildUri($collector->getItemValue('id'));

        $this->logger->info(sprintf('Send PUT request %s with body: %s', $uri, $options['body']));

        try {
            $response = $this->client->request('PUT', $uri, $options);
        } catch (ClientException $exception) {
            throw new RestRequestFailedException(
                self::CLIENT_EXCEPTION_MESSAGE . ' - ' . $exception->getResponse()->getBody()->getContents(),
                $exception->getCode()
            );
        } catch (\Exception $exception) {
            throw new RestRequestFailedException(
                self::CLIENT_EXCEPTION_MESSAGE . ' - ' . $exception->getMessage(),
                $exception->getCode()
            );
        }
        return 204 === $response->getStatusCode();
    }

    /**
     * @param string $code
     * @return string
     */
    private function buildUri($code)
    {
        return sprintf(self::API_UPDATE_URI, $code);
    }
}
