<?php

/**
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2018 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Lifestyle\Sylius\RestApiClientBundle\Services\ProductVariant\Version1\Create;

use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\ClientException;
use JMS\Serializer\SerializerInterface;
use Lifestyle\DataCollector\DataCollectorInterface;
use Lifestyle\Sylius\RestApiClientBundle\Event\ProductVariant\PreSerializeCreateModelEvent;
use Lifestyle\Sylius\RestApiClientBundle\Exception\RestRequestFailedException;
use Psr\Log\LoggerInterface;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;

/**
 * Class Handler
 * @package Lifestyle\Sylius\RestApiClientBundle\Services\ProductVariant\Version1\Create
 */
class Handler
{
    const API_CREATE_URI = '/api/v1/products/%s/variants/';
    const CLIENT_EXCEPTION_MESSAGE = 'Could not add product variant object!';

    /**
     * @var SerializerInterface
     */
    private $serializer;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * @var ClientInterface
     */
    private $client;

    /**
     * @var Mapper
     */
    private $mapper;

    /**
     * @var EventDispatcherInterface
     */
    private $dispatcher;

    /**
     * @param SerializerInterface $serializer
     * @param LoggerInterface $logger
     * @param ClientInterface $client
     * @param Mapper $mapper
     * @param EventDispatcherInterface $dispatcher
     */
    public function __construct(
        SerializerInterface $serializer,
        LoggerInterface $logger,
        ClientInterface $client,
        Mapper $mapper,
        EventDispatcherInterface $dispatcher
    ) {
        $this->serializer = $serializer;
        $this->logger = $logger;
        $this->client = $client;
        $this->mapper = $mapper;
        $this->dispatcher = $dispatcher;
    }

    /**
     * @param DataCollectorInterface $collector
     * @return bool
     * @throws RestRequestFailedException
     * @throws \GuzzleHttp\Exception\GuzzleException
     */
    public function create(DataCollectorInterface $collector)
    {
        if (!strlen($collector->getItemValue('parentId'))) {
            throw new RestRequestFailedException(
                'An unexpected error occurred while saving the object: Parent product code is missing.'
            );
        }

        $product = $this->mapper->mapRequestModel($collector);

        $event = new PreSerializeCreateModelEvent($product, $collector);
        $this->dispatcher->dispatch(PreSerializeCreateModelEvent::NAME, $event);

        $options = [
            'body' => $this->serializer->serialize($product, 'json'),
            'headers' => ['Content-type' => 'application/json']
        ];

        $uri = $this->buildUri($collector->getItemValue('parentId'));
        $this->logger->info(
            sprintf('Send POST request to %s with body: %s', $uri, $options['body'])
        );
        try {
            $response = $this->client->request('POST', $uri, $options);
        } catch (ClientException $exception) {
            throw new RestRequestFailedException(
                self::CLIENT_EXCEPTION_MESSAGE . ' - ' . $exception->getResponse()->getBody()->getContents(),
                $exception->getCode()
            );
        } catch (\Exception $exception) {
            throw new RestRequestFailedException(
                self::CLIENT_EXCEPTION_MESSAGE . ' - ' . $exception->getMessage(),
                $exception->getCode()
            );
        }

        return 201 === $response->getStatusCode();
    }

    /**
     * @param string $parentCode
     * @return string
     */
    private function buildUri($parentCode)
    {
        return sprintf(self::API_CREATE_URI, $parentCode);
    }
}
