<?php

/**
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2017 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Lifestyle\Sylius\RestApiClientBundle\Services\ProductVariant\Version1\Create;

use Lifestyle\DataCollector\DataCollectorInterface;
use Lifestyle\Sylius\RestApiClientBundle\ApiConfiguration\ProductVariantServiceConfiguration;
use Lifestyle\Sylius\RestApiClientBundle\Exception\InvalidConfigurationException;
use Lifestyle\Sylius\RestApiClientBundle\Services\ProductOption\Version1\GeneralModels\Response\ProductOption as ProductOptionResponse;
use Lifestyle\Sylius\RestApiClientBundle\Services\ProductVariant\Version1\Create\Model\Request\ChannelPrice;
use Lifestyle\Sylius\RestApiClientBundle\Services\ProductVariant\Version1\Create\Model\Request\ProductVariantInterface;
use Lifestyle\Sylius\RestApiClientBundle\Services\ProductVariant\Version1\Create\Model\Request\Translation;

/**
 * Class Mapper
 * @package Lifestyle\Sylius\RestApiClientBundle\Services\ProductVariant\Version1\Create
 */
class Mapper
{
    /**
     * @var ProductVariantServiceConfiguration
     */
    private $configuration;

    /**
     * @var string
     */
    private $modelClass;

    /**
     * Mapper constructor.
     * @param ProductVariantServiceConfiguration $configuration
     */
    public function __construct(ProductVariantServiceConfiguration $configuration)
    {
        $this->configuration = $configuration;
    }

    /**
     * @param string $modelClass
     */
    public function setModelClass(string $modelClass)
    {
        $this->modelClass = $modelClass;
    }

    /**
     * @param DataCollectorInterface $collector
     * @return ProductVariantInterface
     * @throws InvalidConfigurationException
     */
    public function mapRequestModel(DataCollectorInterface $collector)
    {
        $product = $this->createModel();
        $product->setCode($collector->getItemValue('id'));
        $product->setOnHand((int)$collector->getItemValue('onHand'));
        $product->setTracked($collector->getItemValue('tracked'));
        $product->setTaxCategory($collector->getItemValue('taxCategory'));
        $product->setHeight($collector->getItemValue('height'));
        $product->setWidth($collector->getItemValue('width'));
        $product->setDepth($collector->getItemValue('depth'));
        $product->setWeight($collector->getItemValue('weight'));
        $product->setVersion($collector->getItemValue('version'));

        if ($collector->hasItem('sku')) {
            $product->setSku($collector->getItemValue('sku'));
        } else {
            // Fallback for BC
            $product->setSku($collector->getItemValue('articleNumber'));
        }

        if ($this->configuration->isEnableShippingCategory()) {
            $product->setShippingCategory($collector->getItemValue('shippingCategory'));
            $product->setShippingRequired((bool)$collector->getItemValue('shippingRequired'));
        }

        if ($collector->hasItem('enabled')) {
            $product->setEnabled((bool)$collector->getItemValue('enabled'));
        }

        $product = $this->mapChannelPrices($collector, $product);
        $product = $this->mapLocalizedFields($collector, $product);
        return $this->mapOptions($collector, $product);
    }

    /**
     * @return ProductVariantInterface
     * @throws InvalidConfigurationException
     */
    private function createModel(): ProductVariantInterface
    {
        $model = new $this->modelClass;

        if (!$model instanceof ProductVariantInterface) {
            throw new InvalidConfigurationException(
                sprintf(
                    'Request model class for creating products must implement %s. %s given.',
                    ProductVariantInterface::class,
                    $this->modelClass
                )
            );
        }

        return $model;
    }

    /**
     * @param DataCollectorInterface $collector
     * @param ProductVariantInterface $product
     * @return ProductVariantInterface
     */
    private function mapOptions(
        DataCollectorInterface $collector,
        ProductVariantInterface $product
    ): ProductVariantInterface {
        if ($collector->hasItem('options')) {
            $options = $collector->getItemValue('options');
            foreach ((is_array($options) ? $options : [$options]) as $optionCode) {
                $optionValues = [];
                if ($collector->hasItem('options_' . $optionCode)) {
                    $optionValues = $collector->getItemValue('options_' . $optionCode);
                }
                foreach ((is_array($optionValues) ? $optionValues : [$optionValues]) as $optionValue) {
                    $product->addOptionValue($optionCode, ProductOptionResponse::buildCode($optionCode, $optionValue));
                }
            }
        }

        return $product;
    }

    /**
     * @param DataCollectorInterface $collector
     * @param ProductVariantInterface $product
     * @return ProductVariantInterface
     */
    private function mapLocalizedFields(
        DataCollectorInterface $collector,
        ProductVariantInterface $product
    ): ProductVariantInterface {
        $locales = $collector->getCollectedLocales();
        foreach ($locales as $locale) {
            $productName = (string)($collector->getItemValue('name', $locale) ?: $collector->getItemValue('id'));
            $translation = new Translation();
            $translation->setLocale($locale);
            $translation->setName($productName);
            $product->addTranslation($translation);
        }

        return $product;
    }

    /**
     * @param DataCollectorInterface $collector
     * @param ProductVariantInterface $product
     * @return ProductVariantInterface
     */
    private function mapChannelPrices(
        DataCollectorInterface $collector,
        ProductVariantInterface $product
    ): ProductVariantInterface {
        if ($collector->hasItem('channelPrices')) {
            foreach ($collector->getItemValue('channelPrices') as $channelCode => $channelPrice) {
                $channelPriceItem = new ChannelPrice();
                $channelPriceItem->setChannel($channelCode);
                $channelPriceItem->setPrice($channelPrice['price']);
                $channelPriceItem->setOriginalPrice($channelPrice['originalPrice']);
                $product->addChannelPricing($channelPriceItem);
            }
        }

        return $product;
    }
}
