<?php

/**
 * Class HandlerTest
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2018 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Lifestyle\Sylius\RestApiClientBundle\Tests\Services\ProductVariant\Version1\GetCollection;

use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\BadResponseException;
use GuzzleHttp\Exception\ClientException;
use JMS\Serializer\SerializerBuilder;
use JMS\Serializer\SerializerInterface;
use Lifestyle\Sylius\RestApiClientBundle\Services\ProductVariant\Version1\GetCollection\Handler;
use Lifestyle\Sylius\RestApiClientBundle\Exception\RestRequestFailedException;
use Lifestyle\Sylius\RestApiClientBundle\Services\ProductVariant\Version1\GetCollection\Model\Response\Response as ServiceResponse;
use Psr\Http\Message\RequestInterface;
use Psr\Log\LoggerInterface;
use PHPUnit\Framework\TestCase;
use GuzzleHttp\Client;
use GuzzleHttp\Handler\MockHandler;
use GuzzleHttp\HandlerStack;
use GuzzleHttp\Psr7\Response;

/**
 * Class HandlerTest
 * @package Lifestyle\Sylius\RestApiClientBundle\Tests\Services\ProductVariant\Version1\GetCollection
 */
class HandlerTest extends TestCase
{
    public function test_get_request_unexpected_client_exception()
    {
        $this->expectException(RestRequestFailedException::class);
        $this->expectExceptionMessage('An unexpected error occurred while fetching the object: Mocked Client Exception');

        $serializer = $this->getMockBuilder(SerializerInterface::class)->getMock();
        $logger = $this->getMockBuilder(LoggerInterface::class)->getMock();

        // Create a mock and queue the responses.
        $mock = new MockHandler([
            new ClientException("Mocked Client Exception", $this->getMockBuilder(RequestInterface::class)->getMock(), new Response(999)),
        ]);

        $handler = HandlerStack::create($mock);
        $client = new Client(['handler' => $handler]);

        $getHandler = new Handler($serializer, $logger, $client);
        $getHandler->getAll('unitTestParentCode123');
    }

    public function test_get_request_unexpected_exception()
    {
        $this->expectException(RestRequestFailedException::class);
        $this->expectExceptionMessage('An unexpected error occurred while fetching the object: unexpected mocked exception');

        $serializer = $this->getMockBuilder(SerializerInterface::class)->getMock();
        $logger = $this->getMockBuilder(LoggerInterface::class)->getMock();

        // Create a mock and queue the responses.
        $mock = new MockHandler([
            new BadResponseException(
                "unexpected mocked exception",
                $this->getMockBuilder(RequestInterface::class)->getMock()
            ),
        ]);

        $handler = HandlerStack::create($mock);
        $client = new Client(['handler' => $handler]);

        $getHandler = new Handler($serializer, $logger, $client);
        $getHandler->getAll('unitTestParentCode123');
    }

    public function test_get_by_code_throws_type_error_exception()
    {
        $this->expectException(\TypeError::class);

        $serializer = $this->getMockBuilder(SerializerInterface::class)->getMock();
        $logger = $this->getMockBuilder(LoggerInterface::class)->getMock();
        $client = $this->getMockBuilder(Client::class)->getMock();

        $getHandler = new Handler($serializer, $logger, $client);
        $getHandler->getAll(null);
    }

    public function test_get_by_code_throws_an_exception()
    {
        $this->expectException(RestRequestFailedException::class);
        $this->expectExceptionMessage('An unexpected error occurred while fetching the object: product-code empty.');

        $serializer = $this->getMockBuilder(SerializerInterface::class)->getMock();
        $logger = $this->getMockBuilder(LoggerInterface::class)->getMock();
        $client = $this->getMockBuilder(Client::class)->getMock();

        $getHandler = new Handler($serializer, $logger, $client);
        $getHandler->getAll('');
    }

    public function test_build_uri()
    {
        $serializer = $this->getMockBuilder(SerializerInterface::class)->getMock();
        $logger = $this->getMockBuilder(LoggerInterface::class)->getMock();
        $client = $this->getMockBuilder(ClientInterface::class)->getMock();

        $updateHandler = new Handler($serializer, $logger, $client);

        $class = new \ReflectionClass(Handler::class);
        $method = $class->getMethod('buildUri');
        $method->setAccessible(true);

        $uri = $method->invoke($updateHandler, 'parentCode1234');
        $this->assertSame('/api/v1/products/parentCode1234/variants', $uri);
    }

    public function test_get_by_code_will_return_response_model()
    {
        $serializer = SerializerBuilder::create()->build();
        $logger = $this->getMockBuilder(LoggerInterface::class)->getMock();

        $responseBody = <<<EOF
{
    "page": 1,
    "limit": 10,
    "pages": 1,
    "total": 2,
    "_links": {
        "self": {
            "href": "\/api\/v1\/products\/MUG_TH\/variants\/?page=1&limit=10"
        },
        "first": {
            "href": "\/api\/v1\/products\/MUG_TH\/variants\/?page=1&limit=10"
        },
        "last": {
            "href": "\/api\/v1\/products\/MUG_TH\/variants\/?page=1&limit=10"
        }
    },
    "_embedded": {
        "items": [
            {
                "id": 331,
                "code": "medium-mug-theme",
                "optionValues": [],
                "position": 0,
                "translations": [],
                "version": 1,
                "tracked": false,
                "channelPricings": [],
                "_links": {
                    "self": {
                        "href": "\/api\/v1\/products\/MUG_TH\/variants\/medium-mug-theme"
                    }
                }
            },
            {
                "id": 332,
                "code": "double-theme-mug",
                "optionValues": [
                    {
                        "name": "Mug type",
                        "code": "mug_type_double"
                    }
                ],
                "position": 1,
                "translations": {
                    "en_US": {
                        "locale": "en_US",
                        "id": 332,
                        "name": "Double Theme Mug"
                    }
                },
                "version": 1,
                "tracked": true,
                "taxCategory": {
                    "id": 3,
                    "code": "other",
                    "name": "Other",
                    "_links": {
                        "self": {
                            "href": "\/api\/v1\/tax-categories\/other"
                        }
                    }
                },
                "shippingCategory": {
                    "id": 1,
                    "code": "default",
                    "name": "Default shipping category",
                    "_links": {
                        "self": {
                            "href": "\/api\/v1\/shipping-categories\/default"
                        }
                    }
                },
                "tracked": false,
                "channelPricings": {
                    "US_WEB": {
                        "channelCode": "US_WEB",
                        "price": 1200
                    }
                },
                "_links": {
                    "self": {
                        "href": "\/api\/v1\/products\/MUG_TH\/variants\/double-theme-mug"
                    }
                }
            }
        ]
    }
}
EOF;

        // Create a mock and queue the responses.
        $mock = new MockHandler([
            new Response(200, [], $responseBody),
        ]);

        $handler = HandlerStack::create($mock);
        $client = new Client(['handler' => $handler]);

        $getHandler = new Handler($serializer, $logger, $client);
        $responseModel = $getHandler->getAll('unitTestParentCode123');

        $this->assertInstanceOf(ServiceResponse::class, $responseModel);
        $this->assertSame(1, $responseModel->getPage());
        $this->assertSame(10, $responseModel->getLimit());
        $this->assertSame(1, $responseModel->getPages());
        $this->assertSame(2, $responseModel->getTotal());

        $this->assertCount(2, $responseModel->getProductVariants());

        $productVariant = $responseModel->getProductVariants()[0];
        $this->assertSame(331, $productVariant->getId());
        $this->assertSame('medium-mug-theme', $productVariant->getCode());
        $this->assertSame(0, $productVariant->getPosition());
        $this->assertSame(1, $productVariant->getVersion());
        $this->assertFalse($productVariant->isTracked());
        $this->assertSame([], $productVariant->getChannelPricings());

        $productVariant = $responseModel->getProductVariants()[1];
        $this->assertSame(332, $productVariant->getId());
        $this->assertSame('double-theme-mug', $productVariant->getCode());
        $this->assertSame(1, $productVariant->getPosition());
        $this->assertSame(1, $productVariant->getVersion());
        $this->assertFalse($productVariant->isTracked());
        $this->assertCount(1, $productVariant->getChannelPricings());

        $channelPricing = $productVariant->getChannelPricings()[0];
        $this->assertSame('US_WEB', $channelPricing->getChannelCode());
        $this->assertSame(1200, $channelPricing->getPrice());
    }
}
