<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Lifestyle\Sylius\RestApiClientBundle\Helper;

use Psr\Http\Message\RequestInterface;
use Psr\SimpleCache\InvalidArgumentException;
use Sainsburys\Guzzle\Oauth2\AccessToken;
use Sainsburys\Guzzle\Oauth2\Middleware\OAuthMiddleware;
use Symfony\Component\Cache\Simple\FilesystemCache;

/**
 * Class OAuthTokenCache
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package Lifestyle\Sylius\RestApiClientBundle\Helper
 */
class OAuthTokenCache
{
    const CACHE_NAMESPACE = 'oauth';
    const CACHE_ACCESS_TOKEN = 'access_token';

    /**
     * @var OAuthMiddleware
     */
    private $oAuthMiddleware;

    /**
     * @var FilesystemCache
     */
    private $cache;

    /**
     * @var bool
     */
    private $cacheLoaded = false;

    /**
     * @var AccessToken
     */
    private $accessToken;

    /**
     * OAuthTokenCache constructor.
     * @param OAuthMiddleware $oAuthMiddleware
     */
    public function __construct(OAuthMiddleware $oAuthMiddleware)
    {
        $this->oAuthMiddleware = $oAuthMiddleware;
        $this->cache = new FilesystemCache(self::CACHE_NAMESPACE);
    }

    /**
     * @return \Closure
     */
    public function loadAccessToken()
    {
        return function (callable $handler) {
            return function (RequestInterface $request, array $options) use ($handler) {

                // Load from filesystem only if there is no access token available
                if (!$this->cacheLoaded && $this->cache->has(self::CACHE_ACCESS_TOKEN)) {

                    $this->cacheLoaded = true;

                    try {
                        $cachedAccessToken = $this->cache->get(self::CACHE_ACCESS_TOKEN);
                    } catch (InvalidArgumentException $exception) {
                        unset($exception);
                    }

                    if ($cachedAccessToken instanceof AccessToken) {
                        $this->oAuthMiddleware->setAccessToken($cachedAccessToken);
                    }
                }

                return $handler($request, $options);
            };
        };
    }

    /**
     * @return \Closure
     */
    public function saveAccessToken()
    {
        return function (callable $handler) {
            return function (RequestInterface $request, array $options) use ($handler) {

                // Save to filesystem if token has been changed and is valid
                $latestAccessToken = $this->oAuthMiddleware->getAccessToken();
                if (
                    $latestAccessToken !== $this->accessToken &&
                    $latestAccessToken instanceof AccessToken
                ) {
                    $this->accessToken = $latestAccessToken;
                    $this->cache->set(
                        self::CACHE_ACCESS_TOKEN,
                        $this->accessToken,
                        $this->accessToken->getExpires()->getTimestamp() - time()
                    );
                }

                return $handler($request, $options);
            };
        };
    }
}
