<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright 2020 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

declare(strict_types=1);

namespace Lifestyle\Sylius\RestApiClientBundle\Services\TierPrice\Version1\Get;

use Exception;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\ClientException;
use GuzzleHttp\Exception\GuzzleException;
use JMS\Serializer\SerializerInterface;
use Lifestyle\Sylius\RestApiClientBundle\Exception\RestRequestFailedException;
use Lifestyle\Sylius\RestApiClientBundle\Services\ProductVariant\Version1\Get\Model\Response\ProductVariant;
use Lifestyle\Sylius\RestApiClientBundle\Services\TierPrice\Version1\Model\Request\TierPriceList;
use Psr\Http\Message\ResponseInterface;
use Psr\Log\LoggerInterface;

/**
 * Class Handler
 * @package Lifestyle\Sylius\RestApiClientBundle\Services\TierPrice\Version1\Get
 */
class Handler
{
    private const API_GET_URI = '/api/tier-prices/price-lists/%s';

    /**
     * @var SerializerInterface
     */
    private $serializer;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * @var ClientInterface
     */
    private $client;

    /**
     * @param SerializerInterface $serializer
     * @param LoggerInterface     $logger
     * @param ClientInterface     $client
     */
    public function __construct(SerializerInterface $serializer, LoggerInterface $logger, ClientInterface $client)
    {
        $this->serializer = $serializer;
        $this->logger = $logger;
        $this->client = $client;
    }

    /**
     * @param $code
     *
     * @return bool
     *
     * @throws GuzzleException
     * @throws RestRequestFailedException
     */
    public function exists($code)
    {
        return false !== $this->getByCode($code);
    }

    /**
     * @param $code
     *
     * @return bool|ProductVariant
     *
     * @throws RestRequestFailedException
     * @throws GuzzleException
     */
    public function getByCode($code)
    {
        if (empty($code)) {
            throw new RestRequestFailedException(
                'An unexpected error occurred while fetching the object: Product or variant code is missing.'
            );
        }

        $uri = $this->buildUri($code);

        $this->logger->info(sprintf('Send GET request %s', $uri));

        try {
            $response = $this->client->request('GET', $uri);
        } catch (ClientException $exception) {
            if (404 === $exception->getResponse()->getStatusCode()) {
                return false;
            }

            throw new RestRequestFailedException(
                sprintf('An unexpected error occurred while fetching the object: %s', $exception->getMessage()),
                $exception->getCode()
            );
        } catch (Exception $exception) {
            throw new RestRequestFailedException(
                sprintf('An unexpected error occurred while fetching the object: %s', $exception->getMessage()),
                $exception->getCode()
            );
        }

        return (200 === $response->getStatusCode()) ? $this->getResponseModel($response) : false;
    }

    /**
     * @param ResponseInterface $response
     *
     * @return ProductVariant | mixed
     */
    private function getResponseModel(ResponseInterface $response)
    {
        return $this->serializer->deserialize($response->getBody()->getContents(), TierPriceList::class, 'json');
    }

    /**
     * @param $code
     *
     * @return string
     */
    private function buildUri($code)
    {
        return sprintf(self::API_GET_URI, $code);
    }
}
