<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright 2020 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

declare(strict_types=1);

namespace Lifestyle\Sylius\RestApiClientBundle\Services\TierPrice\Version1\Update;

use Exception;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\ClientException;
use GuzzleHttp\Exception\GuzzleException;
use JMS\Serializer\SerializerInterface;
use Lifestyle\DataCollector\DataCollectorInterface;
use Lifestyle\Sylius\RestApiClientBundle\Exception\InvalidConfigurationException;
use Lifestyle\Sylius\RestApiClientBundle\Exception\RestRequestFailedException;
use Lifestyle\Sylius\RestApiClientBundle\Services\TierPrice\Version1\Model\Request\TierPriceList;
use Psr\Log\LoggerInterface;

/**
 * Class Handler
 * @package Lifestyle\Sylius\RestApiClientBundle\Services\TierPrice\Version1\Update
 */
class Handler
{
    private const API_UPDATE_URI = '/api/tier-prices/price-lists/%s';
    private const CLIENT_EXCEPTION_MESSAGE = 'Could not update TierPriceList!';

    /**
     * @var SerializerInterface
     */
    private $serializer;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * @var ClientInterface
     */
    private $client;

    /**
     * @param SerializerInterface $serializer
     * @param LoggerInterface     $logger
     * @param ClientInterface     $client
     */
    public function __construct(
        SerializerInterface $serializer,
        LoggerInterface $logger,
        ClientInterface $client
    ) {
        $this->serializer = $serializer;
        $this->logger = $logger;
        $this->client = $client;
    }

    /**
     * @param DataCollectorInterface $collector
     * @return bool
     * @throws RestRequestFailedException
     * @throws GuzzleException
     * @throws InvalidConfigurationException
     */
    public function update(DataCollectorInterface $collector)
    {
        if (empty($collector->getItemValue('id'))) {
            throw new RestRequestFailedException(
                'An unexpected error occurred while updating the object: id is missing.'
            );
        }

        $tierPriceList = new TierPriceList();
        $tierPriceList
            ->setCode((int) $collector->getItemValue('id'))
            ->setName((string) $collector->getItemValue('name'))
            ->setEnabled((int) $collector->getItemValue('enabled'));


        $options = [
            'body' => $this->serializer->serialize($tierPriceList, 'json'),
            'headers' => ['Content-type' => 'application/json'],
        ];

        $uri = $this->buildUri($collector->getItemValue('id'));
        $this->logger->info(sprintf('Send PATCH request %s with body: %s', $uri, $options['body']));
        try {
            $response = $this->client->request('PATCH', $uri, $options);
        } catch (ClientException $exception) {
            throw new RestRequestFailedException(
                self::CLIENT_EXCEPTION_MESSAGE . ' - ' . $exception->getResponse()->getBody()->getContents(),
                $exception->getCode()
            );
        } catch (Exception $exception) {
            throw new RestRequestFailedException(
                self::CLIENT_EXCEPTION_MESSAGE . ' - ' . $exception->getMessage(),
                $exception->getCode()
            );
        }

        return 204 === $response->getStatusCode();
    }

    /**
     * @param $code
     *
     * @return string
     */
    private function buildUri($code)
    {
        return sprintf(self::API_UPDATE_URI, $code);
    }
}
