<?php

/**
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2017 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

declare(strict_types=1);

namespace Lifestyle\Sylius\RestApiClientBundle\Services\ProductVariant\Version1\Update\Model\Request;

use Doctrine\Common\Collections\ArrayCollection;
use JMS\Serializer\Annotation as Serializer;
use Lifestyle\Sylius\RestApiClientBundle\Services\ProductVariant\Version1\Model\Request\BulkPrice;
use Lifestyle\Sylius\RestApiClientBundle\Services\ProductVariant\Version1\Model\Request\StockInformation;

/**
 * Class Product
 * @package Lifestyle\Sylius\RestApiClientBundle\Services\ProductVariant\Version1\Update\Model\Request
 */
class Product implements ProductVariantInterface
{
    /**
     * @var string|null
     */
    private $code;

    /**
     * @var string|null
     */
    private $sku;

    /**
     * @var bool|null
     */
    private $enabled;

    /**
     * @var bool
     */
    private $tracked = false;

    /**
     * @Serializer\SerializedName("onHand")
     * @var int
     */
    private $onHand;

    /**
     * @Serializer\SerializedName("taxCategory")
     * @var string|null
     */
    private $taxCategory;

    /**
     * @Serializer\SerializedName("shippingCategory")
     * @var string|null
     */
    private $shippingCategory;

    /**
     * @Serializer\SerializedName("shippingRequired")
     * @var bool|null
     */
    private $shippingRequired;

    /**
     * @var int
     */
    private $height;

    /**
     * @var int
     */
    private $width;

    /**
     * @var int
     */
    private $depth;

    /**
     * @var int
     */
    private $weight;

    /**
     * @var int
     */
    private $version;

    /**
     * @var ArrayCollection|Translation[]
     */
    private $translations;

    /**
     * @Serializer\SerializedName("channelPricings")
     * @var ArrayCollection|ChannelPrice[]
     */
    private $channelPricing;

    /**
     * @var ArrayCollection|BulkPrice[]
     *
     * @Serializer\SerializedName("tierPrices")
     */
    private $bulkPrices;

    /**
     * @Serializer\SerializedName("optionValues")
     * @var null|ArrayCollection
     */
    private $optionValues;

    /**
     * @Serializer\SerializedName("availabilityIndicator")
     * @var string|null
     */
    private $availabilityIndicator;

    /**
     * @Serializer\SerializedName("concept")
     * @var string|null
     */
    private $concept;

    /**
     * @var StockInformation[]|ArrayCollection
     *
     * @Serializer\SerializedName("stockInformation")
     */
    protected $stockInformation;

    /**
     * Product constructor.
     */
    public function __construct()
    {
        $this->translations = new ArrayCollection();
        $this->channelPricing = new ArrayCollection();
        $this->bulkPrices = new ArrayCollection();
        $this->stockInformation = new ArrayCollection();
    }

    /**
     * @return null|string
     */
    public function getCode()
    {
        return $this->code;
    }

    /**
     * @param null|string $code
     * @return ProductVariantInterface
     */
    public function setCode($code)
    {
        $this->code = $code;

        return $this;
    }

    /**
     * @return string|null
     */
    public function getSku(): ?string
    {
        return $this->sku;
    }

    /**
     * @param string|null $sku
     */
    public function setSku(?string $sku): void
    {
        $this->sku = $sku;
    }

    /**
     * @param bool|null $enabled
     */
    public function setEnabled(?bool $enabled): void
    {
        $this->enabled = $enabled;
    }

    /**
     * @param Translation $translation
     */
    public function addTranslation(Translation $translation)
    {
        $this->translations->set($translation->getLocale(), $translation);
    }

    /**
     * @return ArrayCollection|Translation[]
     */
    public function getTranslations(): ArrayCollection
    {
        return $this->translations;
    }

    /**
     * @param ChannelPrice $pricing
     */
    public function addChannelPricing(ChannelPrice $pricing)
    {
        $this->channelPricing->set($pricing->getChannel(), $pricing);
    }

    /**
     * @return ArrayCollection|ChannelPrice[]
     */
    public function getChannelPricing(): ArrayCollection
    {
        return $this->channelPricing;
    }

    /**
     * @return int
     */
    public function getOnHand()
    {
        return $this->onHand;
    }

    /**
     * @param int $onHand
     * @return ProductVariantInterface
     */
    public function setOnHand($onHand)
    {
        $this->onHand = $onHand;

        return $this;
    }

    /**
     * @return string|null
     */
    public function getTaxCategory(): ?string
    {
        return $this->taxCategory;
    }

    /**
     * @param string|null $taxCategory
     */
    public function setTaxCategory(?string $taxCategory): void
    {
        $this->taxCategory = $taxCategory;
    }

    /**
     * @return string|null
     */
    public function getShippingCategory(): ?string
    {
        return $this->shippingCategory;
    }

    /**
     * @param string|null $shippingCategory
     */
    public function setShippingCategory(?string $shippingCategory): void
    {
        $this->shippingCategory = $shippingCategory;
    }

    /**
     * @return bool|null
     */
    public function getShippingRequired(): ?bool
    {
        return $this->shippingRequired;
    }

    /**
     * @param bool|null $shippingRequired
     */
    public function setShippingRequired(?bool $shippingRequired): void
    {
        $this->shippingRequired = $shippingRequired;
    }

    /**
     * @return bool
     */
    public function isTracked()
    {
        return $this->tracked;
    }

    /**
     * @param bool $tracked
     * @return ProductVariantInterface
     */
    public function setTracked($tracked)
    {
        $this->tracked = $tracked;

        return $this;
    }

    /**
     * @return int
     */
    public function getHeight()
    {
        return $this->height;
    }

    /**
     * @param int $height
     * @return ProductVariantInterface
     */
    public function setHeight($height)
    {
        $this->height = $height;

        return $this;
    }

    /**
     * @return int
     */
    public function getWidth()
    {
        return $this->width;
    }

    /**
     * @param int $width
     * @return ProductVariantInterface
     */
    public function setWidth($width)
    {
        $this->width = $width;

        return $this;
    }

    /**
     * @return int
     */
    public function getDepth()
    {
        return $this->depth;
    }

    /**
     * @param int $depth
     * @return ProductVariantInterface
     */
    public function setDepth($depth)
    {
        $this->depth = $depth;

        return $this;
    }

    /**
     * @return int
     */
    public function getWeight()
    {
        return $this->weight;
    }

    /**
     * @param int $weight
     * @return ProductVariantInterface
     */
    public function setWeight($weight)
    {
        $this->weight = $weight;

        return $this;
    }

    /**
     * @return int
     */
    public function getVersion()
    {
        return $this->version;
    }

    /**
     * @param int $version
     * @return ProductVariantInterface
     */
    public function setVersion($version)
    {
        $this->version = $version;

        return $this;
    }

    /**
     * @return null|ArrayCollection
     */
    public function getOptionValues(): ?ArrayCollection
    {
        return $this->optionValues;
    }

    /**
     * @param string|int $key
     * @param mixed      $value
     * @return ProductVariantInterface
     */
    public function addOptionValue($key, $value): ProductVariantInterface
    {
        if (null === $this->optionValues) {
            $this->optionValues = new ArrayCollection();
        }
        $this->optionValues->set($key, $value);

        return $this;
    }

    /**
     * @return null|string
     */
    public function getAvailabilityIndicator(): ?string
    {
        return $this->availabilityIndicator;
    }

    /**
     * @param null|string $availabilityIndicator
     *
     * @return Product
     */
    public function setAvailabilityIndicator(?string $availabilityIndicator): Product
    {
        $this->availabilityIndicator = $availabilityIndicator;

        return $this;
    }

    /**
     * @return null|string
     */
    public function getConcept(): ?string
    {
        return $this->concept;
    }

    /**
     * @param null|string $concept
     *
     * @return Product
     */
    public function setConcept(?string $concept): Product
    {
        $this->concept = $concept;

        return $this;
    }

    /**
     * @return ArrayCollection|BulkPrice[]
     */
    public function getBulkPrices(): ArrayCollection
    {
        return $this->bulkPrices;
    }

    /**
     * addBulkPrices
     *
     * @param BulkPrice $price
     */
    public function addBulkPrices(BulkPrice $price)
    {
        $this->bulkPrices->add($price);
    }

    /**
     * @return StockInformation[]|ArrayCollection
     */
    public function getStockInformation(): ArrayCollection
    {
        return $this->stockInformation;
    }

    /**
     * addStockInformation
     * @param StockInformation $stockInformation
     */
    public function addStockInformation(StockInformation $stockInformation)
    {
        $this->stockInformation->add($stockInformation);
    }
}
