<?php

/**
 * Class Mapper
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2017 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Lifestyle\Sylius\RestApiClientBundle\Services\Product\Version1\Create;

use Lifestyle\DataCollector\DataCollectorInterface;
use Lifestyle\Sylius\RestApiClientBundle\Services\Product\Version1\Create\Model\Request\Association;
use Lifestyle\Sylius\RestApiClientBundle\Services\Product\Version1\Create\Model\Request\Product;
use Lifestyle\Sylius\RestApiClientBundle\Services\Product\Version1\Create\Model\Request\Translation;
use Lifestyle\Sylius\RestApiClientBundle\Services\Product\Version1\Create\Model\Request\Attributes;

/**
 * Class Mapper
 * @package Lifestyle\Sylius\RestApiClientBundle\Services\Product\Version1\Create
 */
class Mapper
{
    const ASSOCIATION_TYPE_SIMILAR_PRODUCTS = 'similar_products';

    /**
     * @param DataCollectorInterface $collector
     * @return Product
     */
    public function mapRequestModel(DataCollectorInterface $collector)
    {
        $product = new Product();
        $product->setCode($collector->getItemValue('id'));
        $product->setMainTaxon($collector->getItemValue('mainTaxon'));
        $product->setProductTaxon($collector->getItemValue('productTaxon'));
        $product->setChannels(array_values($collector->getItemValue('channels')));
        $product->setEnabled(true);

        $this->addSimilarProducts($collector, $product);

        $locales = $collector->getCollectedLocales();
        foreach ($locales as $locale) {
            $translation = new Translation();

            $this->mapTranslationField($translation, $collector, 'name', $locale);
            $this->mapTranslationField($translation, $collector, 'slug', $locale);
            $this->mapTranslationField($translation, $collector, 'description', $locale);
            $this->mapTranslationField($translation, $collector, 'shortDescription', $locale);
            $this->mapTranslationField($translation, $collector, 'metaKeywords', $locale);
            $this->mapTranslationField($translation, $collector, 'metaDescription', $locale);

            $translation->setLocale($locale);
            $product->addTranslation($translation);
        }

        // create product options if not exist
        if ($collector->hasItem('options') && is_array($collector->getItemValue('options'))) {
            $product->setOptions(array_keys($collector->getItemValue('options')));
        }

        // create product attributes if not exist
        if ($collector->hasItem('attributes') && is_array($collector->getItemValue('attributes'))) {
            $attributes = new Attributes();
            foreach($collector->getItemValue('attributes') as $key => $value) {
                foreach($locales as $locale) {
                    $attributes->setAttribute($key);
                    $attributes->setLocaleCode($locale);
                    $attributes->setValue($value);
                }
            }
            $product->addAttributes($attributes);
        }

        return $product;
    }

    /**
     * @param DataCollectorInterface $collector
     * @param Product $product
     */
    private function addSimilarProducts(DataCollectorInterface $collector, Product $product): void
    {
        if (!$collector->hasItem('similarProducts')) {
            return;
        }

        $similarProducts = $collector->getItemValue('similarProducts');
        $associationValue = is_array($similarProducts) ? implode(',', $similarProducts) : $similarProducts;

        $association = new Association();
        $association
            ->setType(self::ASSOCIATION_TYPE_SIMILAR_PRODUCTS)
            ->setValue($associationValue);

        $product->addAssociation($association);
    }

    /**
     * @param Translation $translation
     * @param DataCollectorInterface $collector
     * @param string $fieldName
     * @param string $lang
     */
    private function mapTranslationField(Translation $translation, DataCollectorInterface $collector, $fieldName, $lang)
    {
        if ($collector->hasItem($fieldName, $lang)) {
            $method = 'set' . ucfirst(strtolower($fieldName));
            $translation->$method($collector->getItemValue($fieldName, $lang));
        }
    }
}
