<?php

/**
 * Class Handler
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2017 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Lifestyle\Sylius\RestApiClientBundle\Services\Product\Version1;

use Lifestyle\DataCollector\DataCollectorInterface;
use Lifestyle\Sylius\RestApiClientBundle\Exception\RestRequestFailedException;
use Lifestyle\Sylius\RestApiClientBundle\Services\HandlerInterface;
use Lifestyle\Sylius\RestApiClientBundle\Services\ProductOption\Version1\Handler as OptionHandler;
use Lifestyle\Sylius\RestApiClientBundle\Services\ProductAttributes\Version1\Handler as AttributesHandler;
use Psr\Log\LoggerInterface;

/**
 * Class Handler
 * @package Lifestyle\Sylius\RestApiClientBundle\Services\Product\Version1
 */
class Handler implements HandlerInterface
{
    const HANDLER_ID = 'Product';

    /**
     * @var Factory
     */
    private $serviceFactory;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * @var OptionHandler
     */
    private $optionHandler;

    /**
     * @var AttributesHandler
     */
    private $attributeHandler;

    /**
     * Handler constructor.
     * @param Factory $serviceFactory
     * @param LoggerInterface $logger
     * @param OptionHandler $optionHandler
     * @param AttributesHandler $attributeHandler
     */
    public function __construct(Factory $serviceFactory, LoggerInterface $logger, OptionHandler $optionHandler, AttributesHandler $attributeHandler)
    {
        $this->serviceFactory = $serviceFactory;
        $this->logger = $logger;
        $this->optionHandler = $optionHandler;
        $this->attributeHandler = $attributeHandler;
    }

    /**
     * @param DataCollectorInterface $collector
     * @return bool
     */
    public function isResponsible(DataCollectorInterface $collector)
    {
        return
            $collector->getItemValue('className') === self::HANDLER_ID &&
            'object' === $collector->getItemValue('type');
    }

    /**
     * @param DataCollectorInterface $collector
     * @throws RestRequestFailedException
     * @throws \GuzzleHttp\Exception\GuzzleException
     */
    public function createOrUpdate(DataCollectorInterface $collector)
    {
        $this->logger->debug('Product Create or Update ...');

        $this->optionHandler->createOrUpdateOptions($collector);
        $this->attributeHandler->createOrUpdateAttributes($collector);

        if ($this->serviceFactory->get()->exists($collector->getItemValue('id'))) {
            // we need to know the number of variants, if we have 0, we have to use POST - request to set options
            // if we have more than 0, we have to use PATCH, which doesn't work with options
            $numberOfVariants = $this->serviceFactory->get()->getNumberOfVariantsByCode($collector->getItemValue('id'));

            $this->logger->debug('Product already exists -> Update ...');
            $this->serviceFactory->update()->update($collector, $numberOfVariants);
        } else {
            $this->logger->debug('Product does not exist yet -> Create ...');
            $this->serviceFactory->create()->create($collector);
        }
    }

    /**
     * @param DataCollectorInterface $collector
     * @throws RestRequestFailedException
     * @throws \GuzzleHttp\Exception\GuzzleException
     */
    public function delete(DataCollectorInterface $collector)
    {
        $this->logger->debug('Product Delete ...');

        if ($this->serviceFactory->get()->exists($collector->getItemValue('id'))) {
            $this->logger->debug('Product exists -> Delete ...');
            $this->serviceFactory->delete()->deleteByCode($collector->getItemValue('id'));
        }
    }
}
