<?php

/**
 * Class Mapper
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2017 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Lifestyle\Sylius\RestApiClientBundle\Services\ProductVariant\Version1\Update;

use Lifestyle\DataCollector\DataCollectorInterface;
use Lifestyle\Sylius\RestApiClientBundle\Exception\InvalidConfigurationException;
use Lifestyle\Sylius\RestApiClientBundle\Services\ProductVariant\Version1\Update\Model\Request\ChannelPrice;
use Lifestyle\Sylius\RestApiClientBundle\Services\ProductVariant\Version1\Update\Model\Request\ProductVariantInterface;
use Lifestyle\Sylius\RestApiClientBundle\Services\ProductVariant\Version1\Update\Model\Request\Translation as VariantTranslation;

/**
 * Class Mapper
 * @package Lifestyle\Sylius\RestApiClientBundle\Services\ProductVariant\Version1\Update
 */
class Mapper
{
    /**
     * @var string
     */
    private $modelClass;

    /**
     * @param string $modelClass
     */
    public function setModelClass(string $modelClass)
    {
        $this->modelClass = $modelClass;
    }

    /**
     * @param DataCollectorInterface $collector
     * @return ProductVariantInterface
     * @throws InvalidConfigurationException
     */
    public function mapRequestModel(DataCollectorInterface $collector)
    {
        $product = $this->createModel();
        $product
            ->setCode($collector->getItemValue('id'))
            ->setOnHand($collector->getItemValue('onHand'))
            ->setTracked($collector->getItemValue('tracked'))
            //->setTaxCategory($collector->getItemValue('taxCategory'))
            ->setHeight($collector->getItemValue('height'))
            ->setWidth($collector->getItemValue('width'))
            ->setDepth($collector->getItemValue('depth'))
            ->setWeight($collector->getItemValue('weight'))
            ->setVersion($collector->getItemValue('version'));

        if ($collector->hasItem('channelPrices')) {
            foreach ($collector->getItemValue('channelPrices') as $channelCode => $channelPrice) {
                $channelPriceItem = new ChannelPrice();
                $channelPriceItem->setChannel($channelCode);
                $channelPriceItem->setPrice($channelPrice['price']);
                $channelPriceItem->setOriginalPrice($channelPrice['originalPrice']);
                $product->addChannelPricing($channelPriceItem);
            }
        }

        $locales = $collector->getCollectedLocales();
        foreach ($locales as $locale) {
            $translation = new VariantTranslation();
            $this->mapTranslationField($translation, $collector, 'name', $locale);
            $translation->setLocale($locale);

            $product->addTranslation($translation);
        }

        // create product optionsValues if not exist
        if ($collector->hasItem('optionValues') && is_array($collector->getItemValue('optionValues'))) {
            foreach ($collector->getItemValue('optionValues') as $optionCode => $optionValue) {
                $product->addOptionValue($optionCode, $optionCode . '_' . $optionValue);
            }
        }

        return $product;
    }

    /**
     * @return ProductVariantInterface
     * @throws InvalidConfigurationException
     */
    private function createModel(): ProductVariantInterface
    {
        $model = new $this->modelClass;

        if (!$model instanceof ProductVariantInterface) {
            throw new InvalidConfigurationException(
                sprintf(
                    'Request model class for updating product variants must implement %s. %s given.',
                    ProductVariantInterface::class,
                    $this->modelClass
                )
            );
        }

        return $model;
    }

    /**
     * @param VariantTranslation $translation
     * @param DataCollectorInterface $collector
     * @param string $fieldName
     * @param string $lang
     */
    private function mapTranslationField(
        VariantTranslation $translation,
        DataCollectorInterface $collector,
        $fieldName,
        $lang
    ) {
        if ($collector->hasItem($fieldName, $lang)) {
            $method = 'set' . ucfirst(strtolower($fieldName));
            $translation->$method($collector->getItemValue($fieldName, $lang));
        }
    }
}
