<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Lifestyle\Sylius\RestApiClientBundle\Helper\Mapper;

use Lifestyle\DataCollector\DataCollectorInterface;
use Lifestyle\Sylius\RestApiClientBundle\Services\ProductOption\Version1\GeneralModels\Response\ProductOption as ProductOptionResponse;
use Lifestyle\Sylius\RestApiClientBundle\Services\ProductOption\Version1\GeneralModels\Translation;
use Lifestyle\Sylius\RestApiClientBundle\Services\ProductOption\Version1\GeneralModels\Value;

/**
 * Trait ProductOptionMapperTrait
 * @package Lifestyle\Sylius\RestApiClientBundle\Helper\Mapper
 */
trait ProductOptionMapperTrait
{
    /**
     * Returns a list of Value items for api-request
     *
     * @param DataCollectorInterface $collector
     * @param string $optionCode
     * @return Value[]
     */
    private function mapOptionValues(DataCollectorInterface $collector, string $optionCode)
    {
        $locales = $collector->getCollectedLocales();

        $values = [];
        foreach ($this->getOptionValues($collector, $optionCode) as $index => $optionValue) {
            // Ignore empty values - not valid
            if (0 === strlen($optionValue)) {
                continue;
            }

            $value = new Value();
            $value->setCode(ProductOptionResponse::buildCode($optionCode, $optionValue));

            foreach ($locales as $locale) {
                $translation = new Translation();
                $translation->setLocale($locale);
                $translation->setValue($this->getTranslatedOptionValue($collector, $optionCode, $index, $locale));
                $value->addTranslation($translation);
            }
            $values[$value->getCode()] = $value;
        }

        return $values;
    }

    /**
     * @param DataCollectorInterface $collector
     * @param string $optionCode
     * @return array
     */
    private function getOptionValues(DataCollectorInterface $collector, string $optionCode): array
    {
        $optionValues = [];
        if ($collector->hasItem('options_' . $optionCode)) {
            $optionValues = $collector->getItemValue('options_' . $optionCode);
        }

        return is_array($optionValues) ? array_values($optionValues) : [$optionValues];
    }

    /**
     * @param DataCollectorInterface $collector
     * @param string $optionCode
     * @param int $index
     * @param string $locale
     * @return string
     */
    private function getTranslatedOptionValue(
        DataCollectorInterface $collector,
        string $optionCode,
        int $index,
        string $locale
    ): string {
        $optionValues = $this->getTranslatedOptionValues($collector, $optionCode, $locale);
        if (isset($optionValues[$index])) {
            return $optionValues[$index];
        }

        return $this->getOptionValues($collector, $optionCode)[$index];
    }

    /**
     * @param DataCollectorInterface $collector
     * @param string $optionCode
     * @return array
     */
    private function getTranslatedOptionValues(DataCollectorInterface $collector, string $optionCode, $locale): array
    {
        $optionValues = [];
        if ($collector->hasItem('options_translated_' . $optionCode, $locale)) {
            $optionValues = $collector->getItemValue('options_translated_' . $optionCode, $locale);
        }

        return is_array($optionValues) ? array_values($optionValues) : [$optionValues];
    }
}
