<?php

/**
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2018 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Lifestyle\Sylius\RestApiClientBundle\Services\ProductOption\Version1\Get;

use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\ClientException;
use JMS\Serializer\SerializerInterface;
use Lifestyle\Sylius\RestApiClientBundle\Exception\RestRequestFailedException;
use Lifestyle\Sylius\RestApiClientBundle\Services\ProductOption\Version1\GeneralModels\Response\ProductOption;
use Psr\Log\LoggerInterface;

/**
 * Class Handler
 * @package Lifestyle\Sylius\RestApiClientBundle\Services\ProductOption\Version1\Get
 */
class Handler
{
    const API_GET_URI = '/api/v1/product-options/%s';

    /**
     * @var SerializerInterface
     */
    private $serializer;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * @var ClientInterface
     */
    private $client;

    /**
     * @param SerializerInterface $serializer
     * @param LoggerInterface $logger
     * @param ClientInterface $client
     */
    public function __construct(SerializerInterface $serializer, LoggerInterface $logger, ClientInterface $client)
    {
        $this->serializer = $serializer;
        $this->logger = $logger;
        $this->client = $client;
    }

    /**
     * @param $optionCode
     * @return bool
     * @throws RestRequestFailedException
     * @throws \GuzzleHttp\Exception\GuzzleException
     */
    public function exists($optionCode)
    {
        return false !== $this->getByCode($optionCode);
    }

    /**
     * @param $optionCode
     * @return bool|ProductOption
     * @throws RestRequestFailedException
     * @throws \GuzzleHttp\Exception\GuzzleException
     */
    public function getByCode($optionCode)
    {
        if (!strlen($optionCode)) {
            throw new RestRequestFailedException(
                'An unexpected error occurred while fetching the object: Product option code is missing.'
            );
        }

        $uri = $this->buildUri($optionCode);
        $this->logger->info(sprintf('Send GET request %s', $uri));
        try {
            $response = $this->client->request('GET', $uri);
            /** @var ProductOption $productOption */
            $productOption = $this->serializer->deserialize(
                $response->getBody()->getContents(),
                ProductOption::class,
                'json'
            );
            return $productOption;
        } catch (ClientException $exception) {
            if (404 === $exception->getResponse()->getStatusCode()) {
                return false;
            }
            throw new RestRequestFailedException(
                sprintf('An unexpected error occurred while fetching the object: %s', $exception->getMessage()),
                $exception->getCode()
            );
        } catch (\Exception $exception) {
            throw new RestRequestFailedException(
                sprintf('An unexpected error occurred while fetching the object: %s', $exception->getMessage()),
                $exception->getCode()
            );
        }
    }

    /**
     * @param string $name
     * @return string
     */
    private function buildUri($name)
    {
        return sprintf(self::API_GET_URI, $name);
    }
}
