<?php

/**
 * Class Handler
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2017 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Lifestyle\Sylius\RestApiClientBundle\Services\ProductVariant\Version1;

use Lifestyle\DataCollector\DataCollectorInterface;
use Lifestyle\Sylius\RestApiClientBundle\Exception\RestRequestFailedException;
use Lifestyle\Sylius\RestApiClientBundle\Services\HandlerInterface;
use Lifestyle\Sylius\RestApiClientBundle\Services\ProductOption\Version1\Handler as OptionHandler;
use Psr\Log\LoggerInterface;

/**
 * Class Handler
 * @package Lifestyle\Sylius\RestApiClientBundle\Services\ProductVariant\Version1
 */
class Handler implements HandlerInterface
{
    const HANDLER_ID = 'Product';

    /**
     * @var Factory
     */
    private $serviceFactory;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * @var OptionHandler
     */
    private $optionHandler;

    /**
     * @param Factory $serviceFactory
     * @param LoggerInterface $logger
     * @param OptionHandler $optionhandler
     */
    public function __construct(Factory $serviceFactory, LoggerInterface $logger, OptionHandler $optionhandler)
    {
        $this->serviceFactory = $serviceFactory;
        $this->logger = $logger;
        $this->optionHandler = $optionhandler;
    }

    /**
     * @param DataCollectorInterface $collector
     * @return bool
     */
    public function isResponsible(DataCollectorInterface $collector)
    {
        return
            $collector->getItemValue('className') === self::HANDLER_ID &&
            'variant' === $collector->getItemValue('type');
    }

    /**
     * @param DataCollectorInterface $collector
     * @throws RestRequestFailedException
     * @throws \GuzzleHttp\Exception\GuzzleException
     */
    public function createOrUpdate(DataCollectorInterface $collector)
    {
        $this->logger->debug('ProductVariant Create or Update ...');

        $responseModel = $this->serviceFactory->get()->getByCode(
            $collector->getItemValue('parentId'),
            $collector->getItemValue('id')
        );

        if ($responseModel) {
            // Set current product variant version from sylius for the next request.
            $collector->addItem('version', $responseModel->getVersion());
            $this->logger->debug('ProductVariant already exists -> Update ...');
            $this->serviceFactory->update()->update($collector);
        } else {
            $this->logger->debug('ProductVariant does not exist yet -> Create ...');
            $this->serviceFactory->create()->create($collector);
        }
    }

    /**
     * @param DataCollectorInterface $collector
     * @throws RestRequestFailedException
     * @throws \GuzzleHttp\Exception\GuzzleException
     */
    public function delete(DataCollectorInterface $collector)
    {
        $this->logger->debug('ProductVariant Delete ...');

        $parentId = $collector->getItemValue('parentId');
        $variantId = $collector->getItemValue('id');

        if ($this->serviceFactory->get()->exists($parentId, $variantId)) {
            $this->logger->debug('ProductVariant exists -> Delete ...');

            $this->serviceFactory->delete()->deleteByCode($parentId, $variantId);
        }
    }
}
