<?php

/**
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2018 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

declare(strict_types=1);

namespace Lifestyle\Sylius\RestApiClientBundle\Services\ProductVariant\Version1\Update;

use Exception;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\ClientException;
use GuzzleHttp\Exception\GuzzleException;
use JMS\Serializer\SerializerInterface;
use Lifestyle\DataCollector\DataCollectorInterface;
use Lifestyle\Sylius\RestApiClientBundle\Exception\InvalidConfigurationException;
use Lifestyle\Sylius\RestApiClientBundle\Exception\RestRequestFailedException;
use Psr\Log\LoggerInterface;
use Lifestyle\Sylius\RestApiClientBundle\Event\ProductVariant\PreSerializeUpdateModelEvent;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;

/**
 * Class Handler
 * @package Lifestyle\Sylius\RestApiClientBundle\Services\ProductVariant\Version1\Update
 */
class Handler
{
    private const API_UPDATE_URI = '/api/v1/products/%s/variants/%s';
    private const CLIENT_EXCEPTION_MESSAGE = 'Could not update product variant object!';

    /**
     * @var SerializerInterface
     */
    private $serializer;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * @var ClientInterface
     */
    private $client;

    /**
     * @var Mapper
     */
    private $mapper;

    /**
     * @var EventDispatcherInterface
     */
    private $dispatcher;

    /**
     * @param SerializerInterface $serializer
     * @param LoggerInterface $logger
     * @param ClientInterface $client
     * @param Mapper $mapper
     * @param EventDispatcherInterface $dispatcher
     */
    public function __construct(
        SerializerInterface $serializer,
        LoggerInterface $logger,
        ClientInterface $client,
        Mapper $mapper,
        EventDispatcherInterface $dispatcher
    ) {
        $this->serializer = $serializer;
        $this->logger = $logger;
        $this->client = $client;
        $this->mapper = $mapper;
        $this->dispatcher = $dispatcher;
    }

    /**
     * @param DataCollectorInterface $collector
     * @return bool
     * @throws RestRequestFailedException
     * @throws GuzzleException
     * @throws InvalidConfigurationException
     */
    public function update(DataCollectorInterface $collector)
    {
        if (empty($collector->getItemValue('parentId')) || empty($collector->getItemValue('id'))) {
            throw new RestRequestFailedException(
                'An unexpected error occurred while updating the object: Product or variant code is missing.'
            );
        }

        $product = $this->mapper->mapRequestModel($collector);

        $event = new PreSerializeUpdateModelEvent($product, $collector);
        $this->dispatcher->dispatch($event);

        $options = [
            'body' => $this->serializer->serialize($product, 'json'),
            'headers' => ['Content-type' => 'application/json'],
        ];

        $uri = $this->buildUri($collector->getItemValue('parentId'), $collector->getItemValue('id'));
        $this->logger->info(sprintf('Send PATCH request %s with body: %s', $uri, $options['body']));
        try {
            $response = $this->client->request('PATCH', $uri, $options);
        } catch (ClientException $exception) {
            throw new RestRequestFailedException(
                self::CLIENT_EXCEPTION_MESSAGE . ' - ' . $exception->getResponse()->getBody()->getContents(),
                $exception->getCode()
            );
        } catch (Exception $exception) {
            throw new RestRequestFailedException(
                self::CLIENT_EXCEPTION_MESSAGE . ' - ' . $exception->getMessage(),
                $exception->getCode()
            );
        }

        return 204 === $response->getStatusCode();
    }

    /**
     * @param string $parentCode
     * @param string $variantCode
     * @return string
     */
    private function buildUri($parentCode, $variantCode)
    {
        return sprintf(self::API_UPDATE_URI, $parentCode, $variantCode);
    }
}
