<?php

/**
 * Class HandlerTest
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2017 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Lifestyle\Sylius\RestApiClientBundle\Tests\Services\Product\Version1\Update;

use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\ClientException;
use JMS\Serializer\SerializerInterface;
use Lifestyle\DataCollector\DataCollector;
use Lifestyle\Sylius\RestApiClientBundle\ApiConfiguration\ProductServiceConfiguration;
use Lifestyle\Sylius\RestApiClientBundle\Services\Product\Version1\Update\Handler;
use Lifestyle\Sylius\RestApiClientBundle\Services\Product\Version1\Update\Mapper;
use Lifestyle\Sylius\RestApiClientBundle\Exception\RestRequestFailedException;
use Lifestyle\Sylius\RestApiClientBundle\Services\Product\Version1\Update\Model\Request\Product;
use Lifestyle\Sylius\RestApiClientBundle\Services\ProductOption\Version1\Handler as OptionHandler;
use Psr\Http\Message\RequestInterface;
use Psr\Log\LoggerInterface;
use PHPUnit\Framework\TestCase;
use GuzzleHttp\Client;
use GuzzleHttp\Handler\MockHandler;
use GuzzleHttp\HandlerStack;
use GuzzleHttp\Psr7\Response;

/**
 * Class HandlerTest
 * @package Lifestyle\Sylius\RestApiClientBundle\Tests\Services\Product\Version1\Update
 */
class HandlerTest extends TestCase
{
    public function testUpdateSuccess()
    {
        $serializer = $this->getMockBuilder(SerializerInterface::class)->getMock();
        $logger = $this->getMockBuilder(LoggerInterface::class)->getMock();
        $productServiceConfigurationMock = $this->getMockBuilder(ProductServiceConfiguration::class)->disableOriginalConstructor()->getMock();
        $mapper = $this->getMockBuilder(Mapper::class)->setConstructorArgs([$productServiceConfigurationMock])->getMock();
        $mapper->method('mapRequestModel')
            ->willReturn($this->getMockBuilder(Product::class)->disableOriginalConstructor()->getMock());
        $optionHandler = $this->getMockBuilder(OptionHandler::class)->disableOriginalConstructor()->getMock();

        // Create a mock and queue the responses.
        $mock = new MockHandler([
            new Response(204, []),
        ]);

        $handler = HandlerStack::create($mock);
        $client = new Client(['handler' => $handler]);

        $collector = new DataCollector();
        $collector->addItem('id', '123456');

        $updateHandler = new Handler($serializer, $logger, $client, $mapper, $optionHandler);
        $this->assertTrue($updateHandler->update($collector, 1));
    }

    public function testUpdateRequestUnexpectedClientException()
    {
        $this->expectException(RestRequestFailedException::class);

        $serializer = $this->getMockBuilder(SerializerInterface::class)->getMock();
        $logger = $this->getMockBuilder(LoggerInterface::class)->getMock();
        $productServiceConfigurationMock = $this->getMockBuilder(ProductServiceConfiguration::class)->disableOriginalConstructor()->getMock();
        $mapper = $this->getMockBuilder(Mapper::class)->setConstructorArgs([$productServiceConfigurationMock])->getMock();
        $optionHandler = $this->getMockBuilder(OptionHandler::class)->disableOriginalConstructor()->getMock();

        // Create a mock and queue the responses.
        $mock = new MockHandler([
            new ClientException("", $this->getMockBuilder(RequestInterface::class)->getMock(), new Response(999)),
        ]);

        $handler = HandlerStack::create($mock);
        $client = new Client(['handler' => $handler]);

        $collector = new DataCollector();
        $collector->addItem('id', '123456');

        $updateHandler = new Handler($serializer, $logger, $client, $mapper, $optionHandler);
        $updateHandler->update($collector);
    }

    public function testUpdateRequestUnexpectedException()
    {
        $this->expectException(RestRequestFailedException::class);
        $this->expectExceptionMessage('Could not update product object! - unexpected mocked exception');

        $serializer = $this->getMockBuilder(SerializerInterface::class)->getMock();
        $logger = $this->getMockBuilder(LoggerInterface::class)->getMock();
        $productServiceConfigurationMock = $this->getMockBuilder(ProductServiceConfiguration::class)->disableOriginalConstructor()->getMock();
        $mapper = $this->getMockBuilder(Mapper::class)->setConstructorArgs([$productServiceConfigurationMock])->getMock();
        $optionHandler = $this->getMockBuilder(OptionHandler::class)->disableOriginalConstructor()->getMock();

        // Create a mock and queue the responses.
        $mock = new MockHandler([
            new \Exception("unexpected mocked exception"),
        ]);

        $handler = HandlerStack::create($mock);
        $client = new Client(['handler' => $handler]);

        $collector = new DataCollector();
        $collector->addItem('id', '123456');

        $updateHandler = new Handler($serializer, $logger, $client, $mapper, $optionHandler);
        $updateHandler->update($collector);
    }

    public function testUpdateNoCodeException()
    {
        $this->expectException(RestRequestFailedException::class);
        $this->expectExceptionMessage('An unexpected error occurred while updating the object: Product code is missing.');

        $serializer = $this->getMockBuilder(SerializerInterface::class)->getMock();
        $logger = $this->getMockBuilder(LoggerInterface::class)->getMock();
        $productServiceConfigurationMock = $this->getMockBuilder(ProductServiceConfiguration::class)->disableOriginalConstructor()->getMock();
        $mapper = $this->getMockBuilder(Mapper::class)->setConstructorArgs([$productServiceConfigurationMock])->getMock();
        $optionHandler = $this->getMockBuilder(OptionHandler::class)->disableOriginalConstructor()->getMock();

        // Create a mock and queue the responses.
        $mock = new MockHandler([]);

        $handler = HandlerStack::create($mock);
        $client = new Client(['handler' => $handler]);

        $collector = new DataCollector();

        $updateHandler = new Handler($serializer, $logger, $client, $mapper, $optionHandler);
        $updateHandler->update($collector);
    }

    public function testBuildUri()
    {
        $serializer = $this->getMockBuilder(SerializerInterface::class)->getMock();
        $logger = $this->getMockBuilder(LoggerInterface::class)->getMock();
        $client = $this->getMockBuilder(ClientInterface::class)->getMock();
        $productServiceConfigurationMock = $this->getMockBuilder(ProductServiceConfiguration::class)->disableOriginalConstructor()->getMock();
        $mapper = $this->getMockBuilder(Mapper::class)->setConstructorArgs([$productServiceConfigurationMock])->getMock();
        $optionHandler = $this->getMockBuilder(OptionHandler::class)->disableOriginalConstructor()->getMock();

        $updateHandler = new Handler($serializer, $logger, $client, $mapper, $optionHandler);

        $class = new \ReflectionClass(Handler::class);
        $method = $class->getMethod('buildUri');
        $method->setAccessible(true);

        $uri = $method->invoke($updateHandler, 'productCode12345');
        $this->assertSame('/api/v1/products/productCode12345', $uri);
    }
}
