<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       https://www.life-style.de
 */

declare(strict_types=1);

namespace Lifestyle\Sylius\Sso\Security\Authentication\Token;

use Lifestyle\Sylius\Sso\Model\SsoApplication;
use Lifestyle\Sylius\Sso\Model\SsoApplicationRole;
use Lifestyle\Sylius\Sso\Model\SsoResponse;
use Symfony\Component\Security\Core\User\UserInterface;

/**
 * Class TokenFactory
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       https://www.life-style.de
 * @package Lifestyle\Sylius\Sso\Security\Authentication\Token
 */
class TokenFactory
{
    /**
     * @param string $applicationName
     * @param SsoResponse $samlResponse
     * @return SamlToken
     */
    public function createFromResponse(string $applicationName, SsoResponse $samlResponse): SamlToken
    {
        $application = $samlResponse->getApplication();
        $user = $samlResponse->getUser();

        $token = new SamlToken($applicationName, $this->mapRoles($application));

        if (null !== $user) {
            $token->setUser((string)$user->getUsername());
            $token->setUserAttribute('userId', (string)$user->getUserId());
            $token->setUserAttribute('username', (string)$user->getUsername());
            $token->setUserAttribute('email', (string)$user->getEmail());
            $token->setUserAttribute('firstName', (string)$user->getFirstName());
            $token->setUserAttribute('lastName', (string)$user->getLastName());
            $token->setUserAttribute('userGuid', (string)$user->getUserGuid());
            $token->setUserAttribute('userIdentifier', (string)$user->getUserIdentifier());
        }

        if (null !== $application) {
            foreach ($application->getAttributes() ?? [] as $attribute) {
                $token->setAttribute($attribute->getName(), $attribute->getValue());
            }
        }

        return $token;
    }

    /**
     * @param SamlToken $samlToken
     * @param UserInterface $user
     * @return SamlToken
     */
    public function createFromToken(SamlToken $samlToken, UserInterface $user): SamlToken
    {
        $token = new SamlToken($samlToken->getApplicationName(), $user->getRoles());
        $token->setUser($user);
        $token->setAuthenticated(true);
        $token->setAttributes($samlToken->getAttributes());

        return $token;
    }

    /**
     * Map roles from SAML response to a flat list
     *
     * @param SsoApplication|null $application
     * @return array
     */
    private function mapRoles(?SsoApplication $application): array
    {
        if (null === $application) {
            return [];
        }
        return array_map(function (SsoApplicationRole $role) {
            return $role->getName();
        }, $application->getRoles());
    }
}
