<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       https://www.life-style.de
 */

declare(strict_types=1);

namespace Lifestyle\Sylius\Sso\Security\Factory;

use Lifestyle\Sylius\Sso\EventListener\LoginRequestListener;
use Lifestyle\Sylius\Sso\Model\SamlResponseMapper;
use Lifestyle\Sylius\Sso\Security\Authentication\Provider\SamlProvider;
use Lifestyle\Sylius\Sso\Security\Authentication\SimpleSamlAuthenticator;
use Lifestyle\Sylius\Sso\Security\Authentication\Token\TokenFactory;
use Lifestyle\Sylius\Sso\Security\Http\EntryPoint\SamlAuthenticationEntryPoint;
use Symfony\Bundle\SecurityBundle\DependencyInjection\Security\Factory\AbstractFactory;
use Symfony\Component\Config\Definition\Builder\NodeDefinition;
use Symfony\Component\DependencyInjection\ChildDefinition;
use Symfony\Component\DependencyInjection\ContainerBuilder;
use Symfony\Component\DependencyInjection\Reference;

/**
 * Class SamlFactory
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       https://www.life-style.de
 * @package Lifestyle\Sylius\Sso\Security\Factory
 */
class SamlFactory extends AbstractFactory
{
    /**
     * @var string
     */
    private $authenticatorId;

    /**
     * Defines the position at which the provider is called.
     * Possible values: pre_auth, form, http, and remember_me.
     *
     * @return string
     */
    public function getPosition()
    {
        return 'pre_auth';
    }

    /**
     * Defines the configuration key used to reference the provider
     * in the firewall configuration.
     *
     * @return string
     */
    public function getKey()
    {
        return 'sso_login';
    }

    /**
     * @inheritDoc
     */
    public function create(ContainerBuilder $container, $id, $config, $userProviderId, $defaultEntryPointId)
    {
        // SimpleSAML authenticator is needed by most of the other services - create first
        $this->authenticatorId = $this->createAuthenticator($container, $id, $config);

        $result = parent::create($container, $id, $config, $userProviderId, $defaultEntryPointId);

        $this->createLoginRequestListener($container, $id, $config);

        return $result;
    }

    /**
     * @param NodeDefinition $node
     */
    public function addConfiguration(NodeDefinition $node)
    {
        parent::addConfiguration($node);

        $node
            ->children()
            ->scalarNode('application_name')->isRequired()->cannotBeEmpty()->end()
            ->scalarNode('service_provider_name')->isRequired()->cannotBeEmpty()->end()
            ->scalarNode('sso_context')->isRequired()->cannotBeEmpty()->end()// admin|shop
            ->end();
    }

    /**
     * Subclasses must return the id of a service which implements the
     * AuthenticationProviderInterface.
     *
     * @param ContainerBuilder $container
     * @param string $id The unique id of the firewall
     * @param array $config The options array for this listener
     * @param string $userProviderId The id of the user provider
     *
     * @return string never null, the id of the authentication provider
     */
    protected function createAuthProvider(ContainerBuilder $container, $id, $config, $userProviderId)
    {
        $providerId = 'lifestyle.sylius.sso.security.authentication.provider.saml_provider.' . $id;
        $container
            ->setDefinition($providerId, new ChildDefinition(SamlProvider::class))
            ->addArgument(new Reference($userProviderId))
            ->addArgument($config['application_name']);

        return $providerId;
    }

    /**
     * @param ContainerBuilder $container
     * @param string $id
     * @param array $config
     * @param string $defaultEntryPoint
     * @return string
     */
    protected function createEntryPoint($container, $id, $config, $defaultEntryPoint)
    {
        $entryPointId = 'lifestyle.sylius.sso.security.http.entry_point.saml_authentication_entry_point.' . $id;
        $container
            ->setDefinition($entryPointId, new ChildDefinition(SamlAuthenticationEntryPoint::class))
            ->addArgument(new Reference($this->authenticatorId));

        return $entryPointId;
    }

    /**
     * @param ContainerBuilder $container
     * @param string $id
     * @param array $config
     * @param string $userProvider
     * @return string
     */
    protected function createListener($container, $id, $config, $userProvider)
    {
        $listenerId = parent::createListener($container, $id, $config, $userProvider);

        $container
            ->getDefinition($listenerId)
            ->addArgument(new Reference(SamlResponseMapper::class))
            ->addArgument(new Reference(TokenFactory::class))
            ->addArgument(new Reference($this->authenticatorId))
            ->addArgument($config['application_name']);

        return $listenerId;
    }

    /**
     * @inheritDoc
     */
    protected function getListenerId()
    {
        return 'lifestyle.sylius.sso.security.firewall.saml_listener';
    }

    /**
     * @param ContainerBuilder $container
     * @param string $id
     * @param array $config
     * @return string
     */
    private function createAuthenticator(ContainerBuilder $container, string $id, array $config): string
    {
        $samlAuthenticatorId = 'lifestyle.sylius.sso.security.authentication.simple_saml_authenticator.' . $id;
        $container
            ->setDefinition($samlAuthenticatorId, new ChildDefinition(SimpleSamlAuthenticator::class))
            ->setPublic(true)
            ->addArgument($config['check_path'])
            ->addArgument($config['default_target_path'])
            ->addArgument($config['service_provider_name'])
            ->addTag('lifestyle.sylius.sso.security.authentication.simple_saml_authenticator.' . $config['sso_context']);

        return $samlAuthenticatorId;
    }

    /**
     * @param ContainerBuilder $container
     * @param string $id
     * @param array $config
     * @return string
     */
    private function createLoginRequestListener(ContainerBuilder $container, string $id, array $config): string
    {
        $loginListenerId = 'lifestyle.sylius.sso.event_listener.login_request_listener.' . $id;
        $container
            ->setDefinition($loginListenerId, new ChildDefinition(LoginRequestListener::class))
            ->addArgument(new Reference($this->authenticatorId))
            ->addArgument($config['login_path'])
            ->addTag('kernel.event_listener', ['event' => 'kernel.request']);

        return $loginListenerId;
    }
}
