<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       https://www.life-style.de
 */

declare(strict_types=1);

namespace Lifestyle\Sylius\Sso\Security\Firewall;

use Lifestyle\Sylius\Sso\Model\SamlResponseMapper;
use Lifestyle\Sylius\Sso\Security\Authentication\SimpleSamlAuthenticator;
use Lifestyle\Sylius\Sso\Security\Authentication\Token\TokenFactory;
use Psr\Log\LoggerInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Security\Core\Authentication\AuthenticationManagerInterface;
use Symfony\Component\Security\Core\Authentication\Token\Storage\TokenStorageInterface;
use Symfony\Component\Security\Core\Exception\AuthenticationException;
use Symfony\Component\Security\Http\Authentication\AuthenticationFailureHandlerInterface;
use Symfony\Component\Security\Http\Authentication\AuthenticationSuccessHandlerInterface;
use Symfony\Component\Security\Http\Firewall\AbstractAuthenticationListener;
use Symfony\Component\Security\Http\HttpUtils;
use Symfony\Component\Security\Http\Session\SessionAuthenticationStrategyInterface;
use Symfony\Contracts\EventDispatcher\EventDispatcherInterface;

/**
 * Class SamlListener
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       https://www.life-style.de
 * @package Lifestyle\Sylius\Sso\Security\Firewall
 */
class SamlListener extends AbstractAuthenticationListener
{
    /**
     * @var SamlResponseMapper
     */
    private $samlResponseMapper;

    /**
     * @var TokenFactory
     */
    private $tokenFactory;

    /**
     * @var SimpleSamlAuthenticator
     */
    private $samlAuthenticator;

    /**
     * @var string
     */
    private $applicationName;

    /**
     * SamlListener constructor.
     * @param TokenStorageInterface $tokenStorage
     * @param AuthenticationManagerInterface $authenticationManager
     * @param SessionAuthenticationStrategyInterface $sessionStrategy
     * @param HttpUtils $httpUtils
     * @param string $providerKey
     * @param AuthenticationSuccessHandlerInterface $successHandler
     * @param AuthenticationFailureHandlerInterface $failureHandler
     * @param array $options
     * @param LoggerInterface $logger
     * @param EventDispatcherInterface $dispatcher
     * @param SamlResponseMapper $samlResponseMapper
     * @param TokenFactory $tokenFactory
     * @param SimpleSamlAuthenticator $samlAuthenticator
     * @param string $applicationName
     */
    public function __construct(
        TokenStorageInterface $tokenStorage,
        AuthenticationManagerInterface $authenticationManager,
        SessionAuthenticationStrategyInterface $sessionStrategy,
        HttpUtils $httpUtils,
        string $providerKey,
        AuthenticationSuccessHandlerInterface $successHandler,
        AuthenticationFailureHandlerInterface $failureHandler,
        array $options = [],
        LoggerInterface $logger,
        EventDispatcherInterface $dispatcher,
        SamlResponseMapper $samlResponseMapper,
        TokenFactory $tokenFactory,
        SimpleSamlAuthenticator $samlAuthenticator,
        string $applicationName
    ) {
        $options = array_merge([
            'username_parameter' => '_username',
            'password_parameter' => '_password',
            'post_only' => false,
        ], $options);
        parent::__construct(
            $tokenStorage,
            $authenticationManager,
            $sessionStrategy,
            $httpUtils,
            $providerKey,
            $successHandler,
            $failureHandler,
            $options,
            $logger,
            $dispatcher
        );

        $this->samlResponseMapper = $samlResponseMapper;
        $this->tokenFactory = $tokenFactory;
        $this->samlAuthenticator = $samlAuthenticator;
        $this->applicationName = $applicationName;
    }

    /**
     * @inheritDoc
     */
    protected function attemptAuthentication(Request $request)
    {
        if (!$this->samlAuthenticator->isAuthenticated()) {
            throw new AuthenticationException('User is not authenticated.');
        }

        // Create token from SAML response
        $samlResponse = $this->samlResponseMapper->mapResponse($this->applicationName, $this->samlAuthenticator);

        $failure = null;
        if (null === $samlResponse->getApplication()) {
            $failure = new AuthenticationException('User is not connected to this application.');
        }
        if (null === $samlResponse->getUser()) {
            $failure = new AuthenticationException('User has not been set in SAML response.');
        }

        $token = $this->tokenFactory->createFromResponse($this->applicationName, $samlResponse);

        if (null !== $failure) {
            $failure->setToken($token);
            throw $failure;
        }

        // Start authentication
        return $this->authenticationManager->authenticate($token);
    }
}
