<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       https://www.life-style.de
 */

declare(strict_types=1);

namespace Lifestyle\Sylius\Sso\Security\Logout;

use Lifestyle\Sylius\Sso\Exception\InvalidConfigurationException;
use Lifestyle\Sylius\Sso\Security\Authentication\SimpleSamlAuthenticator;
use Sylius\Component\Channel\Context\ChannelContextInterface;
use Sylius\Component\Core\Storage\CartStorageInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Security\Http\HttpUtils;
use Symfony\Component\Security\Http\Logout\LogoutSuccessHandlerInterface;

/**
 * Class ShopUserLogoutHandler
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       https://www.life-style.de
 * @package Lifestyle\Sylius\Sso\Security\Logout
 */
class ShopUserLogoutHandler implements LogoutSuccessHandlerInterface
{
    /**
     * @var HttpUtils
     */
    private $httpUtils;

    /**
     * @var SimpleSamlAuthenticator|null
     */
    private $samlAuthenticator;

    /**
     * @var ChannelContextInterface
     */
    private $channelContext;

    /**
     * @var CartStorageInterface
     */
    private $cartStorage;

    /**
     * @var array
     */
    private $options;

    /**
     * ShopUserLogoutHandler constructor.
     * @param HttpUtils $httpUtils
     * @param SimpleSamlAuthenticator|null $samlAuthenticator
     * @param ChannelContextInterface $channelContext
     * @param CartStorageInterface $cartStorage
     */
    public function __construct(
        HttpUtils $httpUtils,
        ?SimpleSamlAuthenticator $samlAuthenticator,
        ChannelContextInterface $channelContext,
        CartStorageInterface $cartStorage
    ) {
        $this->httpUtils = $httpUtils;
        $this->samlAuthenticator = $samlAuthenticator;
        $this->channelContext = $channelContext;
        $this->cartStorage = $cartStorage;
    }

    /**
     * @param array $options
     */
    public function setOptions(array $options): void
    {
        $this->options = $options;
    }

    /**
     * {@inheritdoc}
     */
    public function onLogoutSuccess(Request $request)
    {
        if (null === $this->samlAuthenticator) {
            throw new InvalidConfigurationException(
                'Unable to finish logout process! Invalid firewall configuration. Parameter "sso_context" must be set to "shop".'
            );
        }

        $channel = $this->channelContext->getChannel();
        $this->cartStorage->removeForChannel($channel);

        $target = $this->options['target'] ?: '/';
        if ($this->samlAuthenticator->isAuthenticated()) {
            $target = $this->samlAuthenticator->getLogoutURL($target);
        }

        return $this->httpUtils->createRedirectResponse($request, $target);
    }
}
