<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       https://www.life-style.de
 */

declare(strict_types=1);

namespace Lifestyle\Sylius\Sso\Security\Authentication\Token;

use Lifestyle\Sylius\Sso\Exception\InvalidArgumentException;
use Symfony\Component\Security\Core\Authentication\Token\AbstractToken;

/**
 * Class SamlToken
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       https://www.life-style.de
 * @package Lifestyle\Sylius\Sso\Security\Authentication\Token
 */
class SamlToken extends AbstractToken
{
    /**
     * @var string
     */
    private $applicationName;

    /**
     * @var string
     */
    private $providerKey;

    /**
     * @var array
     */
    private $userAttributes = [];

    /**
     * SamlToken constructor.
     * @param string $applicationName
     * @param string $providerKey
     * @param array $roles
     */
    public function __construct(string $applicationName, string $providerKey, array $roles)
    {
        $this->applicationName = $applicationName;
        $this->providerKey = $providerKey;
        parent::__construct($roles);
    }

    /**
     * @return string
     */
    public function getApplicationName(): string
    {
        return $this->applicationName;
    }

    /**
     * @return string
     */
    public function getProviderKey(): string
    {
        return $this->providerKey;
    }

    /**
     * @param string $name
     * @return bool
     */
    public function hasUserAttribute(string $name): bool
    {
        return \array_key_exists($name, $this->userAttributes);
    }

    /**
     * @param string $name
     * @return mixed
     */
    public function getUserAttribute(string $name)
    {
        if (!\array_key_exists($name, $this->userAttributes)) {
            throw new InvalidArgumentException(sprintf('This token has no "%s" user attribute.', $name));
        }

        return $this->userAttributes[$name];
    }

    /**
     * @param string $name
     * @param mixed $value
     */
    public function setUserAttribute(string $name, $value): void
    {
        $this->userAttributes[$name] = $value;
    }

    /**
     * @return array
     */
    public function __serialize(): array
    {
        return [$this->applicationName, $this->providerKey, $this->userAttributes, parent::__serialize()];
    }

    /**
     * @param array $data
     */
    public function __unserialize(array $data): void
    {
        [$this->applicationName, $this->providerKey, $this->userAttributes, $parentData] = $data;
        parent::__unserialize($parentData);
    }

    /**
     * Returns the user credentials.
     *
     * @return mixed The user credentials
     */
    public function getCredentials()
    {
        return '';
    }
}
