<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       https://www.life-style.de
 */

declare(strict_types=1);

namespace Lifestyle\Sylius\Sso\Security\Factory;

use Lifestyle\Sylius\Sso\EventListener\LoginRequestListener;
use Lifestyle\Sylius\Sso\Security\Authentication\Provider\SamlProvider;
use Lifestyle\Sylius\Sso\Security\Authentication\SimpleSamlAuthenticator;
use Lifestyle\Sylius\Sso\Security\Firewall\SamlListener;
use Lifestyle\Sylius\Sso\Security\Http\EntryPoint\SamlAuthenticationEntryPoint;
use Symfony\Bundle\SecurityBundle\DependencyInjection\Security\Factory\SecurityFactoryInterface;
use Symfony\Component\Config\Definition\Builder\NodeDefinition;
use Symfony\Component\DependencyInjection\ChildDefinition;
use Symfony\Component\DependencyInjection\ContainerBuilder;
use Symfony\Component\DependencyInjection\Reference;

/**
 * Class SamlFactory
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       https://www.life-style.de
 * @package Lifestyle\Sylius\Sso\Security\Factory
 */
class SamlFactory implements SecurityFactoryInterface
{
    private $options = [
        'check_path' => '/login_check',
        'use_forward' => false,
        'require_previous_session' => false,
        'application_name' => '',
        'logout_path' => '/',
    ];

    private $defaultSuccessHandlerOptions = [
        'always_use_default_target_path' => false,
        'default_target_path' => '/',
        'login_path' => '/login',
        'target_path_parameter' => '_target_path',
        'use_referer' => false,
    ];

    private $defaultFailureHandlerOptions = [
        'failure_path' => null,
        'failure_forward' => false,
        'login_path' => '/login',
        'failure_path_parameter' => '_failure_path',
    ];

    /**
     * Defines the position at which the provider is called.
     * Possible values: pre_auth, form, http, and remember_me.
     *
     * @return string
     */
    public function getPosition()
    {
        return 'pre_auth';
    }

    /**
     * Defines the configuration key used to reference the provider
     * in the firewall configuration.
     *
     * @return string
     */
    public function getKey()
    {
        return 'sso_login';
    }

    /**
     * @inheritDoc
     */
    public function create(ContainerBuilder $container, $id, $config, $userProviderId, $defaultEntryPointId)
    {
        $authenticatorId = $this->createAuthenticator($container, $id, $config);
        $authProviderId = $this->createAuthProvider($container, $id, $config, $userProviderId);
        $listenerId = $this->createListener($container, $id, $config, $authenticatorId);
        $entryPointId = $this->createEntryPoint($container, $id, $config, $authenticatorId);
        $this->createLoginRequestListener($container, $id, $config, $authenticatorId);

        return [$authProviderId, $listenerId, $entryPointId];
    }

    /**
     * @param NodeDefinition $node
     */
    public function addConfiguration(NodeDefinition $node)
    {
        $builder = $node->children();

        $builder
            ->scalarNode('provider')->end()
            ->scalarNode('success_handler')->end()
            ->scalarNode('failure_handler')->end()
            ->scalarNode('logout_path')->isRequired()->cannotBeEmpty()->end()
            ->scalarNode('application_name')->isRequired()->cannotBeEmpty()->end()
            ->scalarNode('service_provider_name')->isRequired()->cannotBeEmpty()->end()
            ->scalarNode('sso_context')->isRequired()->cannotBeEmpty()->end(); // admin|shop

        foreach (array_merge($this->options, $this->defaultSuccessHandlerOptions, $this->defaultFailureHandlerOptions) as $name => $default) {
            if (\is_bool($default)) {
                $builder->booleanNode($name)->defaultValue($default);
            } else {
                $builder->scalarNode($name)->defaultValue($default);
            }
        }
    }

    /**
     * Subclasses must return the id of a service which implements the
     * AuthenticationProviderInterface.
     *
     * @param ContainerBuilder $container
     * @param string $id The unique id of the firewall
     * @param array $config The options array for this listener
     * @param string $userProviderId The id of the user provider
     *
     * @return string never null, the id of the authentication provider
     */
    private function createAuthProvider(ContainerBuilder $container, string $id, array $config, string $userProviderId): string
    {
        $providerId = 'lifestyle.sylius.sso.security.authentication.provider.saml_provider.' . $id;
        $container
            ->setDefinition($providerId, new ChildDefinition(SamlProvider::class))
            ->addArgument(new Reference($userProviderId))
            ->addArgument($config['application_name']);

        return $providerId;
    }

    /**
     * @param ContainerBuilder $container
     * @param string $id
     * @param array $config
     * @param string $authenticatorId
     * @return string
     */
    private function createEntryPoint(ContainerBuilder $container, string $id, array $config, string $authenticatorId): string
    {
        $entryPointId = 'lifestyle.sylius.sso.security.http.entry_point.saml_authentication_entry_point.' . $id;
        $container
            ->setDefinition($entryPointId, new ChildDefinition(SamlAuthenticationEntryPoint::class))
            ->addArgument(new Reference($authenticatorId))
            ->addArgument($config['application_name']);

        return $entryPointId;
    }

    /**
     * @param ContainerBuilder $container
     * @param string $id
     * @param array $config
     * @param string $authenticatorId
     * @return string
     */
    private function createListener(ContainerBuilder $container, string $id, array $config, string $authenticatorId): string
    {
        $listenerId = 'lifestyle.sylius.sso.security.firewall.saml_listener.' . $id;
        $container
            ->setDefinition($listenerId, new ChildDefinition(SamlListener::class))
            ->addArgument(new Reference($this->createAuthenticationSuccessHandler($container, $id, $config)))
            ->addArgument(new Reference($this->createAuthenticationFailureHandler($container, $id, $config)))
            ->addArgument(new Reference($authenticatorId))
            ->addArgument($id)
            ->addArgument(array_intersect_key($config, $this->options));

        return $listenerId;
    }

    /**
     * @param ContainerBuilder $container
     * @param string $id
     * @param array $config
     * @return string
     */
    private function createAuthenticator(ContainerBuilder $container, string $id, array $config): string
    {
        $samlAuthenticatorId = 'lifestyle.sylius.sso.security.authentication.simple_saml_authenticator.' . $id;
        $container
            ->setDefinition($samlAuthenticatorId, new ChildDefinition(SimpleSamlAuthenticator::class))
            ->setPublic(true)
            ->addArgument($config['check_path'])
            ->addArgument($config['default_target_path'])
            ->addArgument($config['service_provider_name'])
            ->addTag('lifestyle.sylius.sso.security.authentication.simple_saml_authenticator.' . $config['sso_context']);

        return $samlAuthenticatorId;
    }

    /**
     * @param ContainerBuilder $container
     * @param string $id
     * @param array $config
     * @param string $authenticatorId
     * @return string
     */
    private function createLoginRequestListener(ContainerBuilder $container, string $id, array $config, string $authenticatorId): string
    {
        $loginListenerId = 'lifestyle.sylius.sso.event_listener.login_request_listener.' . $id;
        $container
            ->setDefinition($loginListenerId, new ChildDefinition(LoginRequestListener::class))
            ->addArgument(new Reference($authenticatorId))
            ->addArgument($config['login_path'])
            ->addTag('kernel.event_listener', ['event' => 'kernel.request']);

        return $loginListenerId;
    }

    private function createAuthenticationSuccessHandler(ContainerBuilder $container, string $id, array $config): string
    {
        $successHandlerId = 'security.authentication.success_handler.'.$id.'.'.str_replace('-', '_', $this->getKey());
        $options = array_intersect_key($config, $this->defaultSuccessHandlerOptions);

        if (isset($config['success_handler'])) {
            $successHandler = $container->setDefinition($successHandlerId, new ChildDefinition('security.authentication.custom_success_handler'));
            $successHandler->replaceArgument(0, new Reference($config['success_handler']));
            $successHandler->replaceArgument(1, $options);
            $successHandler->replaceArgument(2, $id);
        } else {
            $successHandler = $container->setDefinition($successHandlerId, new ChildDefinition('security.authentication.success_handler'));
            $successHandler->addMethodCall('setOptions', [$options]);
            $successHandler->addMethodCall('setProviderKey', [$id]);
        }

        return $successHandlerId;
    }

    private function createAuthenticationFailureHandler(ContainerBuilder $container, string $id, array $config): string
    {
        $failureHandlerid ='security.authentication.failure_handler.'.$id.'.'.str_replace('-', '_', $this->getKey());
        $options = array_intersect_key($config, $this->defaultFailureHandlerOptions);

        if (isset($config['failure_handler'])) {
            $failureHandler = $container->setDefinition($failureHandlerid, new ChildDefinition('security.authentication.custom_failure_handler'));
            $failureHandler->replaceArgument(0, new Reference($config['failure_handler']));
            $failureHandler->replaceArgument(1, $options);
        } else {
            $failureHandler = $container->setDefinition($failureHandlerid, new ChildDefinition('security.authentication.failure_handler'));
            $failureHandler->addMethodCall('setOptions', [$options]);
        }

        return $failureHandlerid;
    }
}
