<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       https://www.life-style.de
 */

declare(strict_types=1);

namespace Lifestyle\Sylius\Sso\Security\Firewall;

use Lifestyle\Sylius\Sso\Exception\RuntimeException;
use Lifestyle\Sylius\Sso\Model\SamlResponseMapper;
use Lifestyle\Sylius\Sso\Security\Authentication\SimpleSamlAuthenticator;
use Lifestyle\Sylius\Sso\Security\Authentication\Token\SamlToken;
use Lifestyle\Sylius\Sso\Security\Authentication\Token\TokenFactory;
use Psr\Log\LoggerAwareInterface;
use Psr\Log\LoggerAwareTrait;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Event\RequestEvent;
use Symfony\Component\Security\Core\Authentication\AuthenticationManagerInterface;
use Symfony\Component\Security\Core\Authentication\Token\Storage\TokenStorageInterface;
use Symfony\Component\Security\Core\Authentication\Token\TokenInterface;
use Symfony\Component\Security\Core\Exception\AuthenticationException;
use Symfony\Component\Security\Core\Security;
use Symfony\Component\Security\Http\Authentication\AuthenticationFailureHandlerInterface;
use Symfony\Component\Security\Http\Authentication\AuthenticationSuccessHandlerInterface;
use Symfony\Component\Security\Http\Firewall\LegacyListenerTrait;
use Symfony\Component\Security\Http\Firewall\ListenerInterface;
use Symfony\Component\Security\Http\HttpUtils;
use Symfony\Component\Security\Http\Logout\LogoutSuccessHandlerInterface;
use Symfony\Component\Security\Http\Session\SessionAuthenticationStrategyInterface;

/**
 * Class SamlListener
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       https://www.life-style.de
 * @package Lifestyle\Sylius\Sso\Security\Firewall
 */
class SamlListener implements ListenerInterface, LoggerAwareInterface
{
    use LegacyListenerTrait;
    use LoggerAwareTrait;

    /**
     * @var TokenStorageInterface
     */
    private $tokenStorage;

    /**
     * @var AuthenticationManagerInterface
     */
    private $authenticationManager;

    /**
     * @var SessionAuthenticationStrategyInterface
     */
    private $sessionStrategy;

    /**
     * @var HttpUtils
     */
    private $httpUtils;

    /**
     * @var SamlResponseMapper
     */
    private $samlResponseMapper;

    /**
     * @var TokenFactory
     */
    private $tokenFactory;

    /**
     * @var AuthenticationSuccessHandlerInterface
     */
    private $successHandler;

    /**
     * @var AuthenticationFailureHandlerInterface
     */
    private $failureHandler;

    /**
     * @var SimpleSamlAuthenticator
     */
    private $samlAuthenticator;

    /**
     * @var string
     */
    private $providerKey;

    /**
     * @var array
     */
    private $options;

    /**
     * SamlListener constructor.
     * @param TokenStorageInterface $tokenStorage
     * @param AuthenticationManagerInterface $authenticationManager
     * @param SessionAuthenticationStrategyInterface $sessionStrategy
     * @param HttpUtils $httpUtils
     * @param SamlResponseMapper $samlResponseMapper
     * @param TokenFactory $tokenFactory
     * @param AuthenticationSuccessHandlerInterface $successHandler
     * @param AuthenticationFailureHandlerInterface $failureHandler
     * @param LogoutSuccessHandlerInterface $logoutHandler
     * @param SimpleSamlAuthenticator $samlAuthenticator
     * @param string $providerKey
     * @param array $options
     */
    public function __construct(
        TokenStorageInterface $tokenStorage,
        AuthenticationManagerInterface $authenticationManager,
        SessionAuthenticationStrategyInterface $sessionStrategy,
        HttpUtils $httpUtils,
        SamlResponseMapper $samlResponseMapper,
        TokenFactory $tokenFactory,
        AuthenticationSuccessHandlerInterface $successHandler,
        AuthenticationFailureHandlerInterface $failureHandler,
        SimpleSamlAuthenticator $samlAuthenticator,
        string $providerKey,
        array $options
    ) {
        $this->tokenStorage = $tokenStorage;
        $this->authenticationManager = $authenticationManager;
        $this->sessionStrategy = $sessionStrategy;
        $this->httpUtils = $httpUtils;
        $this->samlResponseMapper = $samlResponseMapper;
        $this->tokenFactory = $tokenFactory;
        $this->successHandler = $successHandler;
        $this->failureHandler = $failureHandler;
        $this->samlAuthenticator = $samlAuthenticator;
        $this->providerKey = $providerKey;
        $this->options = $options;
    }

    /**
     * Handles form based authentication.
     *
     * @param RequestEvent $event
     */
    public function __invoke(RequestEvent $event)
    {
        $request = $event->getRequest();

        $requiresAuthentication = $this->requiresAuthentication($request);
        if (!$request->hasSession()) {
            throw new RuntimeException('This authentication method requires a session.');
        }

        $response = null;
        if ($requiresAuthentication) {
            try {
                if (null === $returnValue = $this->attemptAuthentication()) {
                    return;
                }

                if ($returnValue instanceof TokenInterface) {
                    $this->sessionStrategy->onAuthentication($request, $returnValue);

                    $response = $this->onSuccess($request, $returnValue);
                } elseif ($returnValue instanceof Response) {
                    $response = $returnValue;
                } else {
                    throw new RuntimeException('attemptAuthentication() must either return a Response, an implementation of TokenInterface, or null.');
                }
            } catch (AuthenticationException $e) {
                $response = $this->onFailure($request, $e);
            }
        } elseif ($this->requiresLogout($request)) {
            $response = $this->httpUtils->createRedirectResponse($request, $this->options['logout_path']);
        }

        if (null !== $response) {
            $event->setResponse($response);
        }
    }

    /**
     * @param Request $request
     * @return bool
     */
    private function requiresAuthentication(Request $request): bool
    {
        return $this->httpUtils->checkRequestPath($request, $this->options['check_path']);
    }

    /**
     * @param Request $request
     * @return bool
     */
    private function requiresLogout(Request $request)
    {
        return !$this->samlAuthenticator->isAuthenticated() &&
            ($token = null !== $this->tokenStorage->getToken()) &&
            $token instanceof SamlToken &&
            $token->getProviderKey() === $this->providerKey &&
            isset($this->options['logout_path']) &&
            !$this->httpUtils->checkRequestPath($request, $this->options['logout_path']);
    }

    /**
     * @return TokenInterface
     */
    private function attemptAuthentication(): TokenInterface
    {
        if (!$this->samlAuthenticator->isAuthenticated()) {
            throw new AuthenticationException('User is not authenticated.');
        }

        // Create token from SAML response
        $samlResponse = $this->samlResponseMapper->mapResponse($this->options['application_name'], $this->samlAuthenticator);

        $failure = null;
        if (null === $samlResponse->getApplication()) {
            $failure = new AuthenticationException('User is not connected to this application.');
        }
        if (null === $samlResponse->getUser()) {
            $failure = new AuthenticationException('User has not been set in SAML response.');
        }

        $token = $this->tokenFactory->createFromResponse($this->options['application_name'], $this->providerKey, $samlResponse);

        if (null !== $failure) {
            $failure->setToken($token);
            throw $failure;
        }

        // Start authentication
        return $this->authenticationManager->authenticate($token);
    }

    /**
     * @param Request $request
     * @param TokenInterface $token
     * @return Response
     */
    private function onSuccess(Request $request, TokenInterface $token): Response
    {
        if (null !== $this->logger) {
            $this->logger->info('User has been authenticated successfully.', ['username' => $token->getUsername()]);
        }

        $this->tokenStorage->setToken($token);

        $session = $request->getSession();
        $session->remove(Security::AUTHENTICATION_ERROR);
        $session->remove(Security::LAST_USERNAME);

        $response = $this->successHandler->onAuthenticationSuccess($request, $token);

        if (!$response instanceof Response) {
            throw new RuntimeException('Authentication Success Handler did not return a Response.');
        }

        return $response;
    }

    /**
     * @param Request $request
     * @param AuthenticationException $failed
     * @return Response
     */
    private function onFailure(Request $request, AuthenticationException $failed): Response
    {
        if (null !== $this->logger) {
            $this->logger->info('Authentication request failed.', ['exception' => $failed]);
        }

        $response = $this->failureHandler->onAuthenticationFailure($request, $failed);

        if (!$response instanceof Response) {
            throw new RuntimeException('Authentication Failure Handler did not return a Response.');
        }

        return $response;
    }
}
