<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       https://www.life-style.de
 */

declare(strict_types=1);

namespace Lifestyle\Sylius\Sso\Security\Http\Authentication;

use Psr\Log\LoggerAwareInterface;
use Psr\Log\LoggerAwareTrait;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Security\Core\Exception\AuthenticationException;
use Symfony\Component\Security\Core\Security;
use Symfony\Component\Security\Http\Authentication\AuthenticationFailureHandlerInterface;
use Symfony\Component\Security\Http\HttpUtils;

/**
 * Class AuthenticationFailureHandler
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       https://www.life-style.de
 * @package Lifestyle\Sylius\Sso\Security\Http\Authentication
 */
class AuthenticationFailureHandler implements AuthenticationFailureHandlerInterface, LoggerAwareInterface
{
    use LoggerAwareTrait;

    /**
     * @var HttpUtils
     */
    private $httpUtils;

    /**
     * @var array
     */
    private $options;

    /**
     * AdminAuthenticationFailureHandler constructor.
     * @param HttpUtils $httpUtils
     */
    public function __construct(HttpUtils $httpUtils)
    {
        $this->httpUtils = $httpUtils;
    }

    /**
     * @param array $options
     */
    public function setOptions(array $options): void
    {
        $this->options = $options;
    }

    /**
     * @inheritDoc
     */
    public function onAuthenticationFailure(Request $request, AuthenticationException $exception)
    {
        if ($request->isXmlHttpRequest()) {
            return new JsonResponse(['success' => false, 'message' => $exception->getMessageKey()], 401);
        }

        $request->getSession()->set(Security::AUTHENTICATION_ERROR, $exception);

        $redirectPath = isset($this->options['failure_path']) ? $this->options['failure_path'] : 'lifestyle_sso_admin_user_login_failure';

        $this->logger->debug('Authentication failure, redirect triggered.', ['failure_path' => $redirectPath]);

        return $this->httpUtils->createRedirectResponse($request, $redirectPath);
    }
}
