<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       https://www.life-style.de
 */

declare(strict_types=1);

namespace Lifestyle\Sylius\Sso\Security\Authentication\Provider;

use Doctrine\Common\Persistence\ObjectManager;
use Lifestyle\Sylius\Sso\Security\Authentication\Token\SamlToken;
use Sylius\Bundle\UserBundle\Provider\UsernameOrEmailProvider as BaseUserProvider;
use Sylius\Component\Core\Model\AdminUserInterface as SyliusAdminUserInterface;
use Sylius\Component\Resource\Factory\FactoryInterface;
use Sylius\Component\User\Canonicalizer\CanonicalizerInterface;
use Sylius\Component\User\Repository\UserRepositoryInterface;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\Security\Core\User\UserInterface;
use Webmozart\Assert\Assert;

/**
 * Class AdminUserProvider
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       https://www.life-style.de
 * @package Lifestyle\Sylius\Sso\Security\Authentication\Provider
 */
class AdminUserProvider extends BaseUserProvider implements SamlUserProviderInterface
{
    /** @var FactoryInterface */
    private $userFactory;

    /** @var ObjectManager */
    private $userManager;

    /** @var RequestStack */
    private $requestStack;

    /**
     * AdminUserProvider constructor.
     * @param string $supportedUserClass
     * @param UserRepositoryInterface $userRepository
     * @param FactoryInterface $userFactory
     * @param ObjectManager $userManager
     * @param RequestStack $requestStack
     * @param CanonicalizerInterface $canonicalizer
     */
    public function __construct(
        string $supportedUserClass,
        UserRepositoryInterface $userRepository,
        FactoryInterface $userFactory,
        ObjectManager $userManager,
        RequestStack $requestStack,
        CanonicalizerInterface $canonicalizer
    ) {
        parent::__construct($supportedUserClass, $userRepository, $canonicalizer);
        $this->userFactory = $userFactory;
        $this->userManager = $userManager;
        $this->requestStack = $requestStack;
    }

    /**
     * @param SamlToken $token
     * @return UserInterface
     */
    public function loadUserBySamlToken(SamlToken $token): UserInterface
    {
        $user = $this->findUser($token->getUsername());

        // Fallback to email address
        if (!$user instanceof UserInterface && $token->hasUserAttribute('email')) {
            $user = $this->findUser($token->getUserAttribute('email'));
        }

        if ($user instanceof UserInterface) {
            return $this->updateUserBySamlToken($user, $token);
        }

        return $this->createUserBySamlToken($token);
    }

    /**
     * @param SamlToken $token
     * @return UserInterface
     */
    private function createUserBySamlToken(SamlToken $token): UserInterface
    {
        /** @var SyliusAdminUserInterface $user */
        $user = $this->userFactory->createNew();
        $user->setUsername($token->getUserAttribute('username'));

        // set random password to prevent issue with not nullable field & potential security hole
        $user->setPlainPassword(substr(sha1(mt_rand() . $token->getUserAttribute('userIdentifier')), 0, 10));

        return $this->updateUserBySamlToken($user, $token);
    }

    /**
     * @param UserInterface $user
     * @param SamlToken $token
     * @return UserInterface
     */
    private function updateUserBySamlToken(UserInterface $user, SamlToken $token): UserInterface
    {
        /** @var SyliusAdminUserInterface $user */
        Assert::isInstanceOf($user, SyliusAdminUserInterface::class);

        $user->setFirstName($token->getUserAttribute('firstName'));
        $user->setLastName($token->getUserAttribute('lastName'));
        $user->setEmail($token->getUserAttribute('email'));
        $user->setEnabled(true);
        $user->setLastLogin(new \DateTime());
        $user->setLocaleCode($user->getLocaleCode() ?? $this->getLocale($token));
        $user->setUpdatedAt(new \DateTime());

        $this->updateRoles($user, $token);

        $this->userManager->persist($user);
        $this->userManager->flush();

        return $user;
    }

    /**
     * @param SamlToken $token
     * @return string
     */
    private function getLocale(SamlToken $token): string
    {
        if ($token->hasAttribute('locale')) {
            return $token->getAttribute('locale');
        }
        return $this->requestStack->getMasterRequest()->getLocale();
    }

    /**
     * save roles from saml token if available
     *
     * @param UserInterface $user
     * @param SamlToken     $token
     */
    private function updateRoles(UserInterface $user, SamlToken $token)
    {
        if (is_array($token->getRoleNames()) && count($token->getRoleNames()) > 0) {
            // remove old user roles first
            foreach ($user->getRoles() as $existingRole) {
                if (!in_array($existingRole, $token->getRoleNames())) {
                    $user->removeRole($existingRole);
                }
            }

            // add missing new roles
            foreach ($token->getRoleNames() as $roleName) {
                if (!$user->hasRole($roleName)) {
                    $user->addRole($roleName);
                }
            }
        }
    }
}
