<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       https://www.life-style.de
 */

declare(strict_types=1);

namespace Lifestyle\Sylius\Sso\Security\Authentication\Provider;

use Doctrine\Common\Persistence\ObjectManager;
use Lifestyle\Sylius\Sso\Events\SsoUserEvents;
use Lifestyle\Sylius\Sso\Security\Authentication\Token\SamlToken;
use Sylius\Bundle\UserBundle\Provider\UsernameOrEmailProvider as BaseUserProvider;
use Sylius\Component\Core\Model\CustomerInterface;
use Sylius\Component\Core\Model\ShopUserInterface as SyliusUserInterface;
use Sylius\Component\Core\Repository\CustomerRepositoryInterface;
use Sylius\Component\Resource\Factory\FactoryInterface;
use Sylius\Component\User\Canonicalizer\CanonicalizerInterface;
use Sylius\Component\User\Repository\UserRepositoryInterface;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;
use Symfony\Component\EventDispatcher\GenericEvent;
use Symfony\Component\Security\Core\User\UserInterface;
use Webmozart\Assert\Assert;

/**
 * Class ShopUserProvider
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       https://www.life-style.de
 * @package Lifestyle\Sylius\Sso\Security\Authentication\Provider
 */
class ShopUserProvider extends BaseUserProvider implements SamlUserProviderInterface
{
    /** @var FactoryInterface */
    private $customerFactory;

    /** @var FactoryInterface */
    private $userFactory;

    /** @var ObjectManager */
    private $userManager;

    /** @var CustomerRepositoryInterface */
    private $customerRepository;

    /** @var EventDispatcherInterface */
    private $eventDispatcher;

    /**
     * ShopUserProvider constructor.
     * @param string $supportedUserClass
     * @param FactoryInterface $customerFactory
     * @param UserRepositoryInterface $userRepository
     * @param FactoryInterface $userFactory
     * @param ObjectManager $userManager
     * @param CustomerRepositoryInterface $customerRepository
     * @param CanonicalizerInterface $canonicalizer
     */
    public function __construct(
        string $supportedUserClass,
        FactoryInterface $customerFactory,
        UserRepositoryInterface $userRepository,
        FactoryInterface $userFactory,
        ObjectManager $userManager,
        CustomerRepositoryInterface $customerRepository,
        CanonicalizerInterface $canonicalizer,
        EventDispatcherInterface $eventDispatcher
    ) {
        parent::__construct($supportedUserClass, $userRepository, $canonicalizer);
        $this->customerFactory = $customerFactory;
        $this->userFactory = $userFactory;
        $this->userManager = $userManager;
        $this->customerRepository = $customerRepository;
        $this->eventDispatcher = $eventDispatcher;
    }

    /**
     * @param SamlToken $token
     * @return UserInterface
     */
    public function loadUserBySamlToken(SamlToken $token): UserInterface
    {
        $user = $this->findUser($token->getUsername());

        // Fallback to email address
        if (!$user instanceof UserInterface && $token->hasUserAttribute('email')) {
            $user = $this->findUser($token->getUserAttribute('email'));
        }

        if ($user instanceof UserInterface) {
            return $this->updateUserBySamlToken($user, $token);
        }

        return $this->createUserBySamlToken($token);
    }

    /**
     * @param SamlToken $token
     * @return UserInterface
     */
    private function createUserBySamlToken(SamlToken $token): UserInterface
    {
        /** @var SyliusUserInterface $user */
        $user = $this->userFactory->createNew();

        $canonicalEmail = $this->canonicalizer->canonicalize($token->getUserAttribute('email'));

        /** @var CustomerInterface $customer */
        $customer = $this->customerRepository->findOneBy(['emailCanonical' => $canonicalEmail]);
        if (null === $customer) {
            /** @var CustomerInterface $customer */
            $customer = $this->customerFactory->createNew();
        }

        $user->setCustomer($customer);
        $user->setUsername($token->getUserAttribute('username') ?: $token->getUserAttribute('email'));

        // set random password to prevent issue with not nullable field & potential security hole
        $user->setPlainPassword(substr(sha1(mt_rand() . $token->getUserAttribute('userIdentifier')), 0, 10));
        $user = $this->updateUserBySamlToken($user, $token);

        // dispatch sso user created event
        $this->eventDispatcher->dispatch(new GenericEvent($user), SsoUserEvents::SSO_USER_CREATED);

        return $user;
    }

    /**
     * @param UserInterface $user
     * @param SamlToken $token
     * @return UserInterface
     */
    private function updateUserBySamlToken(UserInterface $user, SamlToken $token): UserInterface
    {
        /** @var SyliusUserInterface $user */
        Assert::isInstanceOf($user, SyliusUserInterface::class);

        $customer = $user->getCustomer();
        $customer->setEmail($token->getUserAttribute('email'));
        $customer->setFirstName($token->getUserAttribute('firstName'));
        $customer->setLastName($token->getUserAttribute('lastName'));

        $user->setEmail($token->getUserAttribute('email'));
        $user->setEnabled(true);
        $user->setLastLogin(new \DateTime());
        $user->setUpdatedAt(new \DateTime());

        $this->userManager->persist($user);
        $this->userManager->flush();

        return $user;
    }
}
