<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       https://www.life-style.de
 */

declare(strict_types=1);

namespace Lifestyle\Sylius\Sso\Security\Http\EntryPoint;

use Lifestyle\Sylius\Sso\Model\SamlResponseMapper;
use Lifestyle\Sylius\Sso\Security\Authentication\SimpleSamlAuthenticator;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException;
use Symfony\Component\Security\Core\Exception\AuthenticationException;
use Symfony\Component\Security\Http\EntryPoint\AuthenticationEntryPointInterface;
use Symfony\Component\Security\Http\HttpUtils;

/**
 * Class SamlAuthenticationEntryPoint
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       https://www.life-style.de
 * @package Lifestyle\Sylius\Sso\Security\Http\EntryPoint
 */
class SamlAuthenticationEntryPoint implements AuthenticationEntryPointInterface
{
    /**
     * @var HttpUtils
     */
    private $httpUtils;

    /**
     * @var SamlResponseMapper
     */
    private $samlResponseMapper;

    /**
     * @var SimpleSamlAuthenticator
     */
    private $samlAuthenticator;

    /**
     * @var string
     */
    private $applicationName;

    /**
     * SamlAuthenticationEntryPoint constructor.
     * @param HttpUtils $httpUtils
     * @param SamlResponseMapper $samlResponseMapper
     * @param SimpleSamlAuthenticator $samlAuthenticator
     * @param string $applicationName
     */
    public function __construct(
        HttpUtils $httpUtils,
        SamlResponseMapper $samlResponseMapper,
        SimpleSamlAuthenticator $samlAuthenticator,
        string $applicationName
    ) {
        $this->httpUtils = $httpUtils;
        $this->samlResponseMapper = $samlResponseMapper;
        $this->samlAuthenticator = $samlAuthenticator;
        $this->applicationName = $applicationName;
    }

    /**
     * @inheritDoc
     */
    public function start(Request $request, AuthenticationException $authException = null)
    {
        $userAuthenticated = $this->samlAuthenticator->isAuthenticated();

        // Do not redirect ajax requests
        if (!$userAuthenticated && $request->isXmlHttpRequest()) {
            throw new AccessDeniedHttpException();
        }

        // Redirect to login page if user is not authenticated
        if (!$userAuthenticated) {
            return $this->httpUtils->createRedirectResponse($request, $this->samlAuthenticator->getLoginURL());
        }

        // Check if user is connected to current application
        $samlResponse = $this->samlResponseMapper->mapResponse($this->applicationName, $this->samlAuthenticator);
        if (null === $samlResponse->getApplication()) {
            throw new AccessDeniedHttpException('User not connected to current application.');
        }

        throw new AuthenticationException(
            'Invalid authentication start! Check your firewall configuration.',
            500,
            $authException
        );
    }
}
