<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       https://www.life-style.de
 */

declare(strict_types=1);

namespace Lifestyle\Sylius\Sso\Security\Authentication\Provider;

use Lifestyle\Sylius\Sso\Security\Authentication\Token\SamlToken;
use Lifestyle\Sylius\Sso\Security\Authentication\Token\TokenFactory;
use Symfony\Component\Security\Core\Authentication\Provider\AuthenticationProviderInterface;
use Symfony\Component\Security\Core\Authentication\Token\TokenInterface;
use Symfony\Component\Security\Core\Exception\AuthenticationException;
use Symfony\Component\Security\Core\Exception\AuthenticationServiceException;
use Symfony\Component\Security\Core\Exception\BadCredentialsException;
use Symfony\Component\Security\Core\Exception\UsernameNotFoundException;
use Symfony\Component\Security\Core\User\UserInterface;

/**
 * Class SamlProvider
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       https://www.life-style.de
 * @package Lifestyle\Sylius\Sso\Security\Authentication\Provider
 */
class SamlProvider implements AuthenticationProviderInterface
{
    /**
     * @var TokenFactory
     */
    private $tokenFactory;

    /**
     * @var SamlUserProviderInterface
     */
    private $userProvider;

    /**
     * @var string
     */
    private $applicationName;

    /**
     * @var bool
     */
    private $hideUserNotFoundExceptions = false;

    /**
     * SamlProvider constructor.
     * @param TokenFactory $tokenFactory
     * @param SamlUserProviderInterface $userProvider
     * @param string $applicationName
     */
    public function __construct(
        TokenFactory $tokenFactory,
        SamlUserProviderInterface $userProvider,
        string $applicationName
    ) {
        $this->tokenFactory = $tokenFactory;
        $this->userProvider = $userProvider;
        $this->applicationName = $applicationName;
    }

    /**
     * Attempts to authenticate a TokenInterface object.
     *
     * @param TokenInterface $token The TokenInterface instance to authenticate
     *
     * @return TokenInterface An authenticated TokenInterface instance, never null
     *
     * @throws AuthenticationException if the authentication fails
     */
    public function authenticate(TokenInterface $token)
    {
        if (!$this->supports($token)) {
            throw new AuthenticationException('The token is not supported by this authentication provider.');
        }

        $username = $token->getUsername();
        if ('' === $username || null === $username) {
            $username = AuthenticationProviderInterface::USERNAME_NONE_PROVIDED;
        }

        try {
            /** @var SamlToken $token */
            $user = $this->userProvider->loadUserBySamlToken($token);
        } catch (UsernameNotFoundException $exception) {
            if ($this->hideUserNotFoundExceptions) {
                throw new BadCredentialsException('Bad credentials.', 0, $exception);
            }
            $exception->setUsername($username);

            throw $exception;
        }

        if (!$user instanceof UserInterface) {
            throw new AuthenticationServiceException('loadUserBySamlToken() must return a UserInterface.');
        }

        return $this->tokenFactory->createFromToken($token, $user);
    }

    /**
     * Checks whether this provider supports the given token.
     *
     * @param TokenInterface $token
     * @return bool true if the implementation supports the Token, false otherwise
     */
    public function supports(TokenInterface $token)
    {
        return
            $token instanceof SamlToken &&
            $token->getApplicationName() === $this->applicationName;
    }
}
