<?php declare(strict_types=1);

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Lifestyle\Sylius\Taxon\Service\TaxonPublish\Handler\Version1;

use Doctrine\ORM\EntityManager;
use Doctrine\ORM\EntityRepository;
use Lifestyle\DataCollector\DataCollectorInterface;
use Lifestyle\Sylius\Taxon\Database\DatabaseTrait;
use Lifestyle\Sylius\Taxon\Entity\Factory;
use Lifestyle\Sylius\Taxon\Entity\Taxon;
use Lifestyle\Sylius\Taxon\Service\TaxonPublish\Handler\TaxonPublishHandlerInterface;
use Lifestyle\Sylius\Taxon\Service\TaxonPublish\RequestData\RequestParserInterface;
use Lifestyle\Sylius\Taxon\Service\TaxonPublish\RequestData\TaxonInterface;
use Sylius\Component\Core\Model\TaxonImage;

/**
 * Class TaxonPublishHandler
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package Lifestyle\Sylius\Taxon\Service\TaxonPublish\Handler\Version1
 */
class TaxonPublishHandler implements TaxonPublishHandlerInterface
{
    use DatabaseTrait;

    /**
     * @var RequestParserInterface
     */
    private $requestParser;

    /**
     * @var EntityManager
     */
    private $taxonManager;

    /**
     * @var Factory
     */
    private $taxonFactory;

    /**
     * TaxonPublishHandler constructor.
     * @param RequestParserInterface $requestParser
     * @param EntityManager $taxonManager
     * @param Factory $taxonFactory
     */
    public function __construct(
        RequestParserInterface $requestParser,
        EntityManager $taxonManager,
        Factory $taxonFactory
    ) {
        $this->requestParser = $requestParser;
        $this->taxonManager = $taxonManager;
        $this->taxonFactory = $taxonFactory;
    }

    /**
     * @param DataCollectorInterface $dataCollector
     * @return bool
     * @throws \Doctrine\ORM\ORMException
     * @throws \Lifestyle\Sylius\Taxon\Exception\DatabaseException
     */
    public function publish(DataCollectorInterface $dataCollector): bool
    {
        $this->taxonManager = $this->getClearedEntityManager($this->taxonManager);
        $taxonRequest = $this->requestParser->parse($dataCollector);

        $taxon = $this->mapTaxon($taxonRequest, $this->findTaxon($taxonRequest->getId()));

        $this->taxonManager->persist($taxon);
        $this->flush($this->taxonManager);

        return true;
    }

    /**
     * @param TaxonInterface $taxonRequest
     * @param Taxon|null $taxon
     * @return Taxon
     */
    private function mapTaxon(TaxonInterface $taxonRequest, ?Taxon $taxon): Taxon
    {
        if (null === $taxon) {
            $taxon = $this->taxonFactory->taxon();
        }

        $taxon->setCode((string)$taxonRequest->getId());

        $parentTaxonId = $taxonRequest->getParentTaxonId();
        if ($parentTaxonId !== $taxonRequest->getId()) {
            $taxon->setParent(null !== $parentTaxonId ? $this->findTaxon($parentTaxonId) : null);
        }

        $this->handleImage($taxonRequest, $taxon);

        $taxon->getTranslations();

        if (null !== $taxonRequest->getTranslations()) {
            foreach ($taxonRequest->getTranslations() as $translation) {
                $taxonTranslation = $taxon->getTranslation($translation->getLocale());
                $taxonTranslation->setLocale($translation->getLocale());
                $taxonTranslation->setName($translation->getName());
                $taxonTranslation->setDescription($translation->getDescription());
                $taxonTranslation->setSlug($translation->getSlug());
                $taxonTranslation->setUrl($translation->getUrl());
                $taxon->addTranslation($taxonTranslation);
            }
        }

        return $taxon;
    }

    /**
     * @param int $taxonId
     * @return Taxon|null
     */
    private function findTaxon(int $taxonId): ?Taxon
    {
        $taxon = $this->taxonManager->getRepository(Taxon::class)->findOneBy(['code' => (string)$taxonId]);
        return $taxon instanceof Taxon ? $taxon : null;
    }

    /**
     * @param TaxonInterface $taxonRequest
     * @param Taxon $taxon
     */
    private function handleImage(TaxonInterface $taxonRequest, Taxon $taxon)
    {

        /**
         * @var TaxonImage $image
         * @var \Lifestyle\Sylius\Taxon\Service\TaxonPublish\RequestData\Version1\Taxon $taxonRequest
         */
        $image = $taxonRequest->getImage();

        if (null === $image) {
            /**
             * remove all
             */
            foreach ($taxon->getImages() as $remove) {
                $taxon->removeImage($remove);
            }

            return;
        }

        /**
         * right now there is only one image
         */
        if ($taxon->hasImages()) {
            $taxon->getImages()->first()->setPath($image->getPath());

            return;
        }

        $taxon->addImage($image);
    }
}
