<?php declare(strict_types=1);

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Lifestyle\Sylius\Taxon\Service\TaxonPublish\RequestData\Version1;

use Lifestyle\DataCollector\DataCollectorInterface;
use Lifestyle\Sylius\Taxon\Service\TaxonPublish\RequestData\RequestParserInterface;
use Lifestyle\Sylius\Taxon\Service\TaxonPublish\RequestData\TaxonInterface;
use Sylius\Component\Core\Model\TaxonImage;
use Symfony\Component\Asset\Packages;
use Symfony\Component\Validator\Exception\ValidatorException;
use Symfony\Component\Validator\Validator\ValidatorInterface;
use Sylius\Component\Resource\Factory\FactoryInterface;

/**
 * Class RequestParser
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package Lifestyle\Sylius\Taxon\Service\TaxonPublish\RequestData\Version1
 */
class RequestParser implements RequestParserInterface
{
    /**
     * @var Factory
     */
    private $factory;

    /**
     * @var ValidatorInterface
     */
    private $validator;

    /**
     * @var FactoryInterface
     */
    private $taxonImageFactory;

    /**
     * @var Packages
     */
    private $packages;

    /**
     * RequestParser constructor.
     * @param Factory $factory
     * @param ValidatorInterface $validator
     * @param FactoryInterface $taxonImageFactory
     * @param Packages $packages
     */
    public function __construct(Factory $factory, ValidatorInterface $validator, FactoryInterface $taxonImageFactory, Packages $packages)
    {
        $this->factory = $factory;
        $this->validator = $validator;
        $this->taxonImageFactory = $taxonImageFactory;
        $this->packages = $packages;
    }

    /**
     * @param DataCollectorInterface $dataCollector
     * @return TaxonInterface
     */
    public function parse(DataCollectorInterface $dataCollector): TaxonInterface
    {
        $taxon = $this->factory->taxon();
        $taxon->setId($dataCollector->getItemValue('id'));
        $taxon->setParentTaxonId($dataCollector->getItemValue('parentTaxonId') ?: null);

        if (!empty($imageId = $dataCollector->getItemValue('imageId'))) {
            /**
             * @var TaxonImage $image
             */
            $image = $this->taxonImageFactory->createNew();

            $path = $this->packages->getUrl((string) $imageId, 'pimcore');
            $image->setPath($path);

            $taxon->setImage($image);
        }

        foreach ($dataCollector->getCollectedLocales() as $locale) {
            $taxonTranslation = $this->factory->taxonTranslation();
            $taxonTranslation->setLocale($locale);
            $taxonTranslation->setName($dataCollector->getItemValue('name', $locale));
            $taxonTranslation->setDescription($dataCollector->getItemValue('description', $locale));
            $taxonTranslation->setSlug($dataCollector->getItemValue('slug', $locale));
            $taxonTranslation->setUrl($dataCollector->getItemValue('url', $locale));
            $taxon->addTranslation($taxonTranslation);
        }

        $violations = $this->validator->validate($taxon);
        if (0 < count($violations)) {
            $errorMessages = [];
            foreach ($violations as $violation) {
                $errorMessages[] = (string)$violation;
            }
            throw new ValidatorException(sprintf(
                'Bad message format: %s',
                join(', ', $errorMessages)
            ));
        }

        return $taxon;
    }
}
