# Life-Style - Symfony RabbitMQ Consumer Bundle

This Symfony bundle connects your Symfony based application to a [RabbitMQ](https://www.rabbitmq.com), a message service using the Advanced Message Queuing Protocol (AMPQ).

The messages provider must use the Life-Style Data-Collector-Structure. More information about the data structure can be found in the [steps-box data-collector-model](https://stash.life-style.de/projects/SB/repos/data-collector-model). You can use the [Life-Style-RabbitMq-Producer-Bundle](https://stash.life-style.de/projects/LCP/repos/lifestylerabbitmqproducerbundle/browse) as a producer.

## Requirements

Symfony Version 3.4 or 4.x

## Installation

The repositories are private. You have to add them to your projects `composer.json`. Make sure you have access to the repositories.

```yaml
{
  "name": "your-project",

  ...

  "repositories": [
    {
      "type": "git",
      "url": "ssh://git@stash.life-style.de:7999/lcp/lifestylerabbitmqconsumerbundle.git"
    },
    {
      "type": "git",
      "url": "ssh://git@stash.life-style.de:7999/sb/data-collector-model.git"
    }
  ]
}
```

Install the package.

```bash
composer require lifestyle/symfony-rabbitmq-consumer-bundle
```

## Configuration

The bundle uses the [RabbitMqBundle](https://github.com/php-amqplib/RabbitMqBundle) in the background. Formerly known as the OldSoundRabbitMQBundle. This has to be configured. Please refer to the RabbitMqBundles documentation for further details.

The `callback` option needs to be set to the bundles consumer service-id `Lifestyle\RabbitMq\Consumer\Consumer\Consumer`, the `queue_options` name attribute should point to queue's name. The consumers name, `product_data` in this example, can be any valid string. It should be unique in your configuration.

```yaml
old_sound_rabbit_mq:
    connections:
        default:
            url: 'amqp://guest:guest_pw@localhost:5672?vhost=internal'
    consumers:
        product_data:
            connection: default
            callback: Lifestyle\RabbitMq\Consumer\Consumer\Consumer
            queue_options: { name: 'product-data-queue' }
            enable_logger: true
            auto_setup_fabric: false
```

## Message Handler

To receive messages, you have to create your own message handler. The handler must implement the `Lifestyle\RabbitMq\Consumer\Handler\HandlerInterface` and has to be tagged with `lifestyle.rabbit_mq.consumer`. The services can be prioritized.

```yaml
services:
  Your\Message\Handler:
    tags:
     - { name: lifestyle.rabbit_mq.consumer }

  Your\PriorityMessage\Handler:
    tags:
     - { name: lifestyle.rabbit_mq.consumer, priority: 42 }
```

If the `handle`-function returns true, the message is marked as successfully processed. If it returns false, the next message handler will be called. Throw an exception, if an error occurs.

*Example:*

```php
<?php

namespace MyNamespace;

use Lifestyle\RabbitMq\Consumer\Handler\HandlerInterface;

class MyHandler implements HandlerInterface
{
    /**
     * @param DataCollectorInterface $dataCollector
     * @return bool Should return true, if data has been processed
     * @throws ExceptionInterface
     */
    public function handle(DataCollectorInterface $dataCollector): bool
    {
        if (!$this->isResponsible($dataCollector)) {
            return false;
        }

        // Process message
        ...

        return true;
    }

    /**
     * @param DataCollectorInterface $dataCollector
     * @return bool
     */
    private function isResponsible(DataCollectorInterface $dataCollector): bool
    {
        return $dataCollector->getItemValue('event') === 'my_event';
    }
}
```

```yaml
services:
  MyNamespace\MyHandler:
    tags:
     - { name: lifestyle.rabbit_mq.consumer }
```
