<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Lifestyle\RabbitMq\Consumer\Consumer;

use JMS\Serializer\Exception\Exception as SerializerExceptionInterface;
use Lifestyle\DataCollector\DataCollector;
use Lifestyle\DataCollector\DataCollectorInterface;
use Lifestyle\RabbitMq\Consumer\Configuration\ConsumerConfiguration;
use Lifestyle\RabbitMq\Consumer\Exception\ExceptionInterface;
use Lifestyle\RabbitMq\Consumer\Exception\InvalidArgumentException;
use Lifestyle\RabbitMq\Consumer\Processor\Processor;
use OldSound\RabbitMqBundle\RabbitMq\ConsumerInterface;
use PhpAmqpLib\Message\AMQPMessage;
use Psr\Log\LoggerInterface;
use JMS\Serializer\SerializerInterface;
use Symfony\Component\HttpFoundation\Response;

/**
 * Class Consumer
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package Lifestyle\RabbitMq\Consumer\Consumer
 */
class Consumer implements ConsumerInterface
{
    /**
     * @var ConsumerConfiguration
     */
    private $configuration;

    /**
     * @var Processor
     */
    private $processor;

    /**
     * @var SerializerInterface
     */
    private $serializer;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * Consumer constructor.
     * @param ConsumerConfiguration $configuration
     * @param Processor $processor
     * @param SerializerInterface $serializer
     * @param LoggerInterface $logger
     */
    public function __construct(
        ConsumerConfiguration $configuration,
        Processor $processor,
        SerializerInterface $serializer,
        LoggerInterface $logger
    ) {
        $this->configuration = $configuration;
        $this->processor = $processor;
        $this->serializer = $serializer;
        $this->logger = $logger;
    }

    /**
     * @param AMQPMessage $message
     * @return bool
     */
    public function execute(AMQPMessage $message)
    {
        $messageBody = (string)$message->body;

        try {
            $dataCollector = $this->deserialize($messageBody);
            if ($this->processor->execute($dataCollector)) {
                return ConsumerInterface::MSG_ACK;
            }
            $this->logger->error(sprintf(
                'No message handler found for message: %s ...',
                $this->stripMessage($messageBody)
            ));
        } catch (\Exception $exception) {
            $this->logger->log(
                $exception->getCode() < 500 ? 'error' : 'critical',
                sprintf(
                    'Error while processing message: %s - Message: %s',
                    $exception->getMessage(),
                    $this->stripMessage($messageBody)
                ),
                ['exception' => $exception]
            );
        }

        return ConsumerInterface::MSG_REJECT;
    }

    /**
     * @param string $messageBody
     * @return DataCollectorInterface
     * @throws ExceptionInterface
     */
    private function deserialize(string $messageBody): DataCollectorInterface
    {
        try {
            return $this->serializer->deserialize($messageBody, DataCollector::class, $this->configuration->getRequestFormat());
        } catch (SerializerExceptionInterface $exception) {
            throw new InvalidArgumentException(
                'Cannot read message! Deserialization failed.',
                Response::HTTP_BAD_REQUEST,
                $exception
            );
        }
    }

    /**
     * @param string $message
     * @return string
     */
    private function stripMessage(string $message): string
    {
        return preg_replace('/[\s]/','', substr($message, 0, 400));
    }
}
