<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Lifestyle\RabbitMq\Producer\Handler;

use JMS\Serializer\SerializerInterface;
use JMS\Serializer\Exception\Exception as SerializerExceptionInterface;
use Lifestyle\DataCollector\DataCollectorInterface;
use Lifestyle\RabbitMq\Producer\Configuration\RabbitMqHandlerConfiguration;
use Lifestyle\RabbitMq\Producer\Exception\ExceptionInterface;
use Lifestyle\RabbitMq\Producer\Exception\InvalidArgumentException;
use OldSound\RabbitMqBundle\RabbitMq\ProducerInterface;
use Psr\Log\LoggerInterface;

/**
 * Class RabbitMqHandler
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package Lifestyle\RabbitMq\Producer\Handler
 */
class RabbitMqHandler implements HandlerInterface
{
    /**
     * @var RabbitMqHandlerConfiguration
     */
    private $configuration;

    /**
     * @var ProducerInterface
     */
    private $producer;

    /**
     * @var SerializerInterface
     */
    private $serializer;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * RabbitMqHandler constructor.
     * @param RabbitMqHandlerConfiguration $configuration
     * @param ProducerInterface $producer
     * @param SerializerInterface $serializer
     * @param LoggerInterface $logger
     */
    public function __construct(
        RabbitMqHandlerConfiguration $configuration,
        ProducerInterface $producer,
        SerializerInterface $serializer,
        LoggerInterface $logger
    )
    {
        $this->configuration = $configuration;
        $this->producer = $producer;
        $this->serializer = $serializer;
        $this->logger = $logger;
    }

    /**
     * @param DataCollectorInterface $dataCollector
     * @return bool Should return true, if data has been processed
     */
    public function handle(DataCollectorInterface $dataCollector): bool
    {
        if (!$this->isResponsible($dataCollector)) {
            return false;
        }

        try {
            $message = $this->serialize($dataCollector);
        } catch (ExceptionInterface $exception) {
            $this->logger->error($exception->getMessage());
            return false;
        }

        $this->producer->publish(
            $message,
            $this->configuration->getRoutingKey(),
            $this->configuration->getMessageOptions()
        );

        $this->logger->debug('Message has been sent to rabbit-mq: ' . substr($message, 0, 255) . '...');

        return true;
    }

    /**
     * @param DataCollectorInterface $dataCollector
     * @return bool
     */
    private function isResponsible(DataCollectorInterface $dataCollector)
    {
        return
            $dataCollector->hasItem('event') &&
            in_array($dataCollector->getItem('event')->getValue(), $this->configuration->getResponsibleEvents());
    }

    /**
     * @throws ExceptionInterface
     * @param DataCollectorInterface $dataCollector
     * @return string
     */
    private function serialize(DataCollectorInterface $dataCollector): string
    {
        try {
            return $this->serializer->serialize($dataCollector, $this->configuration->getRequestFormat());
        } catch (SerializerExceptionInterface $exception) {
            throw new InvalidArgumentException(
                'Cannot create message! Serialization failed.',
                $exception->getCode(),
                $exception
            );
        }
    }
}
