<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Lifestyle\RabbitMq\Producer\DependencyInjection;

use Lifestyle\RabbitMq\Producer\Adapter\RabbitMqProducer;
use Lifestyle\RabbitMq\Producer\Configuration\RabbitMqHandlerConfiguration;
use Lifestyle\RabbitMq\Producer\Handler\RabbitMqHandler;
use Symfony\Component\DependencyInjection\ChildDefinition;
use Symfony\Component\DependencyInjection\Definition;
use Symfony\Component\HttpKernel\DependencyInjection\Extension;
use Symfony\Component\DependencyInjection\Loader\XmlFileLoader;
use Symfony\Component\DependencyInjection\ContainerBuilder;
use Symfony\Component\Config\FileLocator;
use Symfony\Component\DependencyInjection\Reference;

/**
 * Class LifestyleSymfonyRabbitMqProducerExtension
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */
class LifestyleRabbitMqProducerExtension extends Extension
{
    /**
     * Loads a specific configuration.
     *
     * @throws \InvalidArgumentException When provided tag is not defined in this extension
     */
    public function load(array $configs, ContainerBuilder $container)
    {
        $configuration = $this->getConfiguration($configs, $container);
        $config = $this->processConfiguration($configuration, $configs);

        (new XmlFileLoader($container, new FileLocator(dirname(__DIR__) . '/Resources/config')))->load('services.xml');

        foreach ($config['handler'] as $handlerConfig) {
            $this->configureRabbitMqHandlerConfiguration($handlerConfig, $container);
            $this->configureRabbitMqProducer($handlerConfig, $container);
            $this->configureRabbitMqHandler($handlerConfig, $container);
        }
    }

    /**
     * Add configured producer from old-sound-rabbit-mq-bundle to rabbit-mq-handler
     *
     * @param array $config
     * @param ContainerBuilder $container
     */
    private function configureRabbitMqProducer(array $config, ContainerBuilder $container)
    {
        $serviceId =
            'lifestyle.symfony.rabbit_mq.rabbit_mq_producer.' .
            $config['producer'];

        $container
            ->setDefinition($serviceId, new ChildDefinition(RabbitMqProducer::class))
            ->addArgument($config['producer']);
    }

    /**
     * Add configured producer from old-sound-rabbit-mq-bundle to rabbit-mq-handler
     *
     * @param array $config
     * @param ContainerBuilder $container
     */
    private function configureRabbitMqHandler(array $config, ContainerBuilder $container)
    {
        if ($container->has(RabbitMqHandler::class)) {
            return;
        }

        $serviceId =
            'lifestyle.symfony.rabbit_mq.handler.rabbit_mq_handler.' .
            $config['producer'];

        $configurationServiceId =
            'lifestyle.symfony.rabbit_mq.configuration.rabbit_mq_handler_configuration.' .
            $config['producer'];

        $producerServiceId =
            'lifestyle.symfony.rabbit_mq.rabbit_mq_producer.' .
            $config['producer'];

        $definition = new Definition(RabbitMqHandler::class, [
            new Reference($configurationServiceId),
            new Reference($producerServiceId),
            new Reference('jms_serializer'),
            new Reference('logger'),
        ]);
        $definition->addTag('lifestyle.rabbit_mq.producer');
        $definition->setPublic(false);

        $container->setDefinition($serviceId, $definition);
    }

    /**
     * @param array $config
     * @param ContainerBuilder $container
     */
    private function configureRabbitMqHandlerConfiguration(array $config, ContainerBuilder $container)
    {
        if ($container->has(RabbitMqHandlerConfiguration::class)) {
            return;
        }

        $serviceId =
            'lifestyle.symfony.rabbit_mq.configuration.rabbit_mq_handler_configuration.' .
            $config['producer'];

        $definition = new Definition(RabbitMqHandlerConfiguration::class, [
            $config['request_format'],
            $config['routing_key'],
            $config['message_options'],
            $config['events'],
        ]);
        $definition->setPublic(false);

        $container->setDefinition($serviceId, $definition);
    }
}
