<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright   2019 Lifestyle Webconsulting GmbH
 * @link        http://www.life-style.de
 */

namespace LifeStyle\Symfony\TranslationConsumerBundle\Handler;

use Lifestyle\DataCollector\DataCollectorInterface;
use Lifestyle\RabbitMq\Consumer\Handler\HandlerInterface;
use LifeStyle\Symfony\TranslationConsumerBundle\Configuration\Config;
use LifeStyle\Symfony\TranslationConsumerBundle\Model\TranslationItem;
use LifeStyle\Symfony\TranslationConsumerBundle\Processor\TranslationsProcessor;
use Symfony\Component\Validator\Exception\InvalidOptionsException;
use Symfony\Component\Validator\Validator\ValidatorInterface;

/**
 * Class TranslationHandler
 * @package LifeStyle\Symfony\TranslationConsumerBundle\Handler
 */
class TranslationsHandler implements HandlerInterface
{
    /**
     * @var Config
     */
    private $config;

    /**
     * @var TranslationsProcessor
     */
    private $translationsProcessor;

    /**
     * @var ValidatorInterface
     */
    private $validator;

    /**
     * TranslationsHandler constructor.
     * @param Config $config
     * @param TranslationsProcessor $translationsProcessor
     * @param ValidatorInterface $validator
     */
    public function __construct(
        Config $config,
        TranslationsProcessor $translationsProcessor,
        ValidatorInterface $validator
    ) {
        $this->config = $config;
        $this->translationsProcessor = $translationsProcessor;
        $this->validator = $validator;
    }

    /**
     * @param DataCollectorInterface $dataCollector
     * @return bool Should return true, if data has been processed
     */
    public function handle(DataCollectorInterface $dataCollector): bool
    {
        if (!$this->isResponsible($dataCollector)) {
            return false;
        }

        $selector = $dataCollector->getItemValue('selector');
        $domain = $dataCollector->getItemValue('domain');

        //validate locale
        foreach ($dataCollector->getCollectedLocales() as $locale) {
            try {
                $transValue = $dataCollector->getItemValue('translation', $locale);

                $translationItem = $this->getNewTranslationItem($domain, $locale, $selector, $transValue);
                $errors = $this->validator->validate($translationItem);

                if (is_array($errors) && count($errors) > 0) {
                    throw new InvalidOptionsException((string)$errors);
                }

                $this->translationsProcessor->process($translationItem);

            } catch (\Exception $exception) {
                //@todo handle error
            }
        }

        return true;
    }

    /**
     * @param DataCollectorInterface $dataCollector
     * @return bool
     */
    private function isResponsible(DataCollectorInterface $dataCollector): bool
    {
        return $dataCollector->getItemValue('event') === $this->config->getEventName();
    }

    /**
     * @return TranslationItem
     */
    private function getNewTranslationItem($domain, $locale, $selector, $transValue)
    {
        return new TranslationItem($domain, $locale, $selector, $transValue);
    }
}
