<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright   2019 Lifestyle Webconsulting GmbH
 * @link        http://www.life-style.de
 */

namespace LifeStyle\Symfony\TranslationConsumerBundle\Processor;

use LifeStyle\Symfony\TranslationConsumerBundle\Configuration\Config;
use LifeStyle\Symfony\TranslationConsumerBundle\Helper\FileHelper;
use LifeStyle\Symfony\TranslationConsumerBundle\Helper\YamlHelper;
use LifeStyle\Symfony\TranslationConsumerBundle\Model\TranslationItem;

/**
 * Class TranslationsProcessor
 * @package LifeStyle\Symfony\TranslationConsumerBundle\Processor
 */
class TranslationsProcessor implements TranslationsProcessorInterface
{
    /**
     * @var YamlHelper
     */
    private $yamlHelper;

    /**
     * @var FileHelper
     */
    private $fileHelper;

    /**
     * @var Config
     */
    private $config;

    /**
     * TranslationsProcessor constructor.
     * @param YamlHelper $yamlHelper
     * @param FileHelper $fileHelper
     * @param Config $config
     */
    public function __construct(YamlHelper $yamlHelper, FileHelper $fileHelper, Config $config)
    {
        $this->yamlHelper = $yamlHelper;
        $this->fileHelper = $fileHelper;
        $this->config = $config;
    }

    /**
     * @param TranslationItem $translationItem
     * @return bool
     */
    public function process(TranslationItem $translationItem)
    {
        $transFileName = $translationItem->getYamlFileName();

        //first check if the translations path exists
        if (!$this->fileHelper->getFileSystem()->exists($this->config->getTranslationPath())) {
            $this->fileHelper->getFileSystem()->mkdir($this->config->getTranslationPath());
        }

        //next check if the translation file exists
        $transFileNameWithPath = $this->config->getTranslationPath() . '/' . $transFileName;

        if (!$this->fileHelper->getFileSystem()->exists($transFileNameWithPath)) {
            $this->fileHelper->getFileSystem()->touch($transFileNameWithPath);
        }

        //now get array from yaml parser
        $yamlContentArray = $this->yamlHelper->parse($transFileNameWithPath);
        //add value
        $yamlContentArray[$translationItem->getSelector()] = $translationItem->getTransValue();

        //ok good to go write the file complete new
        $yamlContentString = $this->yamlHelper->createYamlContent($yamlContentArray);
        $this->fileHelper->getFileSystem()->dumpFile($transFileNameWithPath, $yamlContentString);
    }
}
