<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright   2019 Lifestyle Webconsulting GmbH
 * @link        http://www.life-style.de
 */

namespace Lifestyle\Symfony\TranslationConsumerBundle\Handler;

use Lifestyle\DataCollector\DataCollectorInterface;
use Lifestyle\RabbitMq\Consumer\Handler\HandlerInterface;
use Lifestyle\Symfony\TranslationConsumerBundle\Configuration\Config;
use Lifestyle\Symfony\TranslationConsumerBundle\Exception\InvalidArgumentException;
use Lifestyle\Symfony\TranslationConsumerBundle\Helper\FileHelper;
use Lifestyle\Symfony\TranslationConsumerBundle\Model\TranslationItem;
use Lifestyle\Symfony\TranslationConsumerBundle\Processor\TranslationsProcessor;
use Symfony\Component\Validator\Validator\ValidatorInterface;
use Symfony\Component\Filesystem\Exception\IOExceptionInterface;

/**
 * Class TranslationsHandler
 * @package Lifestyle\Symfony\TranslationConsumerBundle\Handler
 */
class TranslationsHandler implements HandlerInterface
{
    /**
     * @var Config
     */
    private $config;

    /**
     * @var TranslationsProcessor
     */
    private $translationsProcessor;

    /**
     * @var ValidatorInterface
     */
    private $validator;

    /**
     * @var FileHelper
     */
    private $fileHelper;

    /**
     * TranslationsHandler constructor.
     * @param Config $config
     * @param TranslationsProcessor $translationsProcessor
     * @param ValidatorInterface $validator
     * @param FileHelper $fileHelper
     */
    public function __construct(
        Config $config,
        TranslationsProcessor $translationsProcessor,
        ValidatorInterface $validator,
        FileHelper $fileHelper
    ) {
        $this->config = $config;
        $this->translationsProcessor = $translationsProcessor;
        $this->validator = $validator;
        $this->fileHelper = $fileHelper;
    }

    /**
     * @param DataCollectorInterface $dataCollector
     * @return bool Should return true, if data has been processed
     * @throws InvalidArgumentException
     */
    public function handle(DataCollectorInterface $dataCollector): bool
    {
        if (!$this->isResponsible($dataCollector)) {
            return false;
        }

        $selector = $dataCollector->getItemValue('selector');
        $domain = $dataCollector->hasItem('domain') ? $dataCollector->getItemValue('domain') : $dataCollector->getItemValue('Domain');

        //validate locale
        foreach ($dataCollector->getCollectedLocales() as $locale) {
            $transValue = $dataCollector->getItemValue('translation', $locale);

            $translationItem = $this->getNewTranslationItem($domain, $locale, $selector, $transValue);
            $errors = $this->validator->validate($translationItem);

            if (is_array($errors) && count($errors) > 0) {
                throw new InvalidArgumentException((string)$errors);
            }

            $this->translationsProcessor->process($translationItem);
        }

        //last but not least delete translation caches
        $this->deleteTranslationCaches();

        return true;
    }

    /**
     * @param DataCollectorInterface $dataCollector
     * @return bool
     */
    private function isResponsible(DataCollectorInterface $dataCollector): bool
    {
        return $dataCollector->getItemValue('event') === $this->config->getEventName();
    }

    /**
     * @return TranslationItem
     */
    private function getNewTranslationItem($domain, $locale, $selector, $transValue)
    {
        return new TranslationItem($domain, $locale, $selector, $transValue);
    }

    /**
     * @throws IOExceptionInterface
     * @return void
     */
    private function deleteTranslationCaches()
    {
        foreach ($this->config->getTranslationCacheDirs() as $path) {
            try {
                $this->fileHelper->getFileSystem()->remove($path);
            } catch (IOExceptionInterface $exception) {
                unset($exception);
            }
        }
    }
}
