<?php

/**
 * Class Index
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\Webservices\ObjectBundle\Database\Data\TreeObject;

use Doctrine\Bundle\DoctrineBundle\Registry as DoctrineRegistry;
use Doctrine\ORM\EntityManager;
use Sso\Webservices\ObjectBundle\Entity\Object as EntityObject;
use Sso\Webservices\ObjectBundle\Database\Repository\Factory as RepositoryFactory;

/**
 * Class Index
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package    Sso\Webservices\ObjectBundle\Database\Data\Object
 */
class Index
{

    /**
     * Doctrine Registry
     *
     * @var DoctrineRegistry
     */
    protected $doctrineRegistry;

    /**
     * @var EntityManager
     */
    protected $entityManager;

    /**
     * @var RepositoryFactory
     */
    protected $repositoryFactory;

    /**
     * @var bool
     */
    protected $fullResponse;

    /**
     * @param DoctrineRegistry $doctrineRegistry
     * @param RepositoryFactory $repoFactory
     */
    public function __construct(DoctrineRegistry $doctrineRegistry, RepositoryFactory $repoFactory)
    {
        $this->doctrineRegistry = $doctrineRegistry;
        $this->entityManager = $this->doctrineRegistry->getManager('Webservice');
        $this->repositoryFactory = $repoFactory;
        $this->fullResponse = false;
    }

    /**
     * Persist and flush entity
     *
     * @param EntityObject $entity
     */
    public function save(EntityObject $entity)
    {
        $this->entityManager->persist($entity);
        $this->entityManager->flush($entity);
    }

    /**
     * Remove and flush entity
     *
     * @param EntityObject $entity
     */
    public function delete(EntityObject $entity)
    {
        $this->entityManager->remove($entity);
        $this->entityManager->flush($entity);
    }

    /**
     * Find all entities
     *
     * @return array
     */
    public function findAll()
    {
        return $this->repositoryFactory->object()->findAll();
    }

    /**
     * Find by method
     *
     * @param array $criteria
     * @param array $orderBy
     * @param int $limit
     * @param int $offset
     * @return Dto\Object[]
     */
    public function findBy(array $criteria, array $orderBy = null, $limit = null, $offset = null)
    {
        $qOrder = ' ORDER BY o.treeId ASC ';
        $qWhere = '';
        $qParams = [];

        if (null === $orderBy) {
            $orderBy = ['treeId ASC'];
        }

        if (null !== $orderBy) {
            foreach ($orderBy as $value) {
                if ('' == $qOrder) {
                    $qOrder .= " ORDER BY ";
                } else {
                    $qOrder .= ", ";
                }
                $qOrder .= " o.$value ";
            }
        }
        $i = 0;
        foreach ($criteria as $key => $value) {
            $i++;
            $cName = ':' . trim(strtolower($key)) . $i;
            if ('' == $qWhere) {
                $qWhere .= " WHERE ";
            } else {
                $qWhere .= " OR ";
            }
            $qWhere .= " o.$key = $cName ";
            $qParams[$cName] = $value;
        }

        $queryFull = '';
        $queryMinimal = 'SELECT o.guid, o.name, o.treeId, o.referenceId, o.parentGuid, ot.id typeId, ot.name typeName, ot.typeOrder typeOrder ' .
            'FROM Sso\Webservices\ObjectBundle\Entity\Object o JOIN o.type ot' . $qWhere . $qOrder;
        $queryString = $this->fullResponse ? $queryFull : $queryMinimal;

        $query = $this->entityManager->createQuery($queryString);

        foreach ($qParams as $key => $value) {
            $query->setParameter($key, $value);
        }

        if ((null !== $limit) && (null !== $offset)) {
            $query->setFirstResult($offset);
            $query->setMaxResults($limit);
        }

        $queryResultSet = $query->getArrayResult();

        $resultSet = [];
        foreach ($queryResultSet as $queryResult) {
            $model = new Dto\Object();
            $model->setGuid($queryResult['guid']);
            $model->setName($queryResult['name']);
            $model->setParentGuid($queryResult['parentGuid']);
            $model->setTreeId($queryResult['treeId']);
            $model->setTypeId($queryResult['typeId']);
            $model->setTypeName($queryResult['typeName']);
            $model->setTypeOrder($queryResult['typeOrder']);
            $model->setReferenceId($queryResult['referenceId']);

            $resultSet[] = $model;
            unset($model);
        }

        return $resultSet;
    }

    public function generationRequired()
    {
        $queryString = "SELECT o.treeId FROM Sso\Webservices\ObjectBundle\Entity\Object o WHERE o.treeId IS NULL OR o.treeId = ''";
        $query = $this->entityManager->createQuery($queryString);

        $query->setFirstResult(0);
        $query->setMaxResults(1);
        $result = $query->getOneOrNullResult();

        return $result !== null;
    }

    /**
     * Find by method
     *
     * @param array $criteria
     * @param array $orderBy
     * @param int $limit
     * @param int $offset
     * @return array
     */
    public function findAllForIdGen()
    {
        $query = $this->entityManager->createQuery('SELECT o.guid, o.parentGuid, o.treeId FROM Sso\Webservices\ObjectBundle\Entity\Object o');
        return $query->getArrayResult();
    }

    public function updateTreeId($guid, $treeId)
    {
        $query = $this->entityManager->createQuery('UPDATE Sso\Webservices\ObjectBundle\Entity\Object o SET o.treeId = :treeId WHERE o.guid = :guid');
        $query->setParameter('treeId', $treeId);
        $query->setParameter('guid', $guid);
        $query->execute();
    }

    /**
     * Get count of all existing entities
     *
     * @return int
     */
    public function getCount()
    {
        return $this->repositoryFactory->object()->createQueryBuilder('b')
            ->select('count(b.guid)')
            ->getQuery()
            ->getSingleScalarResult();
    }

    /**
     * @param integer $full
     * @return $this
     */
    public function setFullResponse($full)
    {
        $this->fullResponse = $full == 1;
        return $this;
    }

}
