<?php

/**
 * Class Show
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\Webservices\ObjectBundle\Model\Request\Language\Show;

use LifeStyle\Tools\WebserviceBundle\Error\Type\External as ExternalError;
use LifeStyle\Tools\WebserviceBundle\Validator\Xml as XmlValidator;
use LifeStyle\Tools\WebserviceBundle\Exception\InvalidRecordException;
use Sso\ApiBundle\Api\Manager as ApiManager;
use Sso\Webservices\ObjectBundle\Entity\Language as EntityLanguage;

/**
 * Class Show
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package    Sso\Webservices\ObjectBundle\Model\Request\Language\Show
 */
class Show
{

    /**
     * @var ApiManager
     */
    protected $apiM;


    /**
     * @var Entity
     */
    protected $helperModel;

    /**
     * @var EntityLanguage[]
     */
    protected $languageModel;

    /**
     * @param ApiManager $apiM
     */
    public function __construct(ApiManager $apiM)
    {
        $this->apiM = $apiM;
    }

    /**
     * Returns a xml-string with allowed structure
     *
     * @return string
     */
    protected function _xmlAllowedElements()
    {
        return '<LanguageType required="true">'
            . '<Code/>'
            . '</LanguageType>';
    }

    /**
     * Init model
     *
     * @param \SimpleXMLElement $simpleXML
     */
    public function init(\SimpleXMLElement $simpleXML)
    {
        $validator = new XmlValidator();
        if (!$validator->validate($simpleXML, $validator->strToXml($this->_xmlAllowedElements()))) {
            $this->apiM->errors()->addErrors($validator->errors()->getErrors());
            throw new InvalidRecordException('Invalid xml elements.');
        }

        // set helper model
        $this->helperModel = $this->apiM->serializer()->deserialize($simpleXML->LanguageType->saveXML(),
            'Sso\Webservices\ObjectBundle\Model\Request\Language\Show\Entity', 'xml');
        $this->validateEntity();

        $this->getLanguageEntities();
    }

    /**
     * @return array
     */
    public function getLanguageEntities()
    {
        // data
        $code = $this->helperModel->getId();

        if (!empty($code)) {
            $languageResult = $this->apiM->database()->webservice()->language()->getLanguageById($code);
            if ($languageResult) {
                $newName = $this->apiM->translator()->language()->getLanguageName($languageResult->getId(), $code);
                $languageResult->setName($newName);

                $this->languageModel = [$languageResult];
            }

            return true;
        }

        $this->languageModel = $this->apiM->database()->webservice()->language()->getLanguageRepository()->findAll();
        return true;
    }

    /**
     * @return EntityLanguage[]
     */
    public function getLanguageModel()
    {
        return $this->languageModel;
    }

    /**
     * Validate entity
     *
     * @param array $validationGroups
     * @return bool
     */
    protected function validateEntity($validationGroups = null)
    {
        $errors = $this->apiM->validator()->validate($this->helperModel, $validationGroups);
        foreach ($errors as $error) {
            list($errorCode, $shortMessage, $longMessage) = explode('##', $error->getMessage());
            $this->apiM->errors()->addError(new ExternalError('v001', $errorCode, $shortMessage, $longMessage));
        }
        if (0 < count($errors)) {
            throw new InvalidRecordException('Invalid xml values');
        }
        return true;
    }

    /**
     * @return int
     */
    public function getLanguagesCount()
    {
        $repo = $this->apiM->database()->webservice()->language()->getLanguageRepository();

        return $repo->createQueryBuilder('b')
            ->select('count(b.id)')
            ->getQuery()
            ->getSingleScalarResult();
    }

}
