<?php

/**
 * Class Show
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2015 Lifestyle Webconsulting GmbH
 * @version    $Id:$
 * @link       http://www.life-style.de
 *
 */

namespace Sso\Webservices\ObjectBundle\Model\Request\Object\ShowMulti;

use LifeStyle\Tools\WebserviceBundle\Error\Type\External as ExternalError;
use LifeStyle\Tools\WebserviceBundle\Validator\Xml as XmlValidator;
use LifeStyle\Tools\WebserviceBundle\Exception\InvalidRecordException;
use Sso\ApiBundle\Api\Manager as ApiManager;
use Sso\Webservices\ObjectBundle\Entity\Object as EntityObject;
use Sso\Webservices\ObjectBundle\Entity\TypeMapping as EntityTypeMapping;
use \Doctrine\DBAL\Types\Type;
use \Doctrine\ORM\EntityRepository;

/**
 * type add request model
 */
class Show
{

    /**
     * @var ApiManager
     */
    protected $apiM;

    /**
     * @var Entity
     */
    protected $helperModel;

    /**
     *
     * @var EntityObject[]
     */
    protected $objectModels;

    /**
     * @var \Sso\Webservices\ObjectBundle\Database\Manager
     */
    protected $objectDbM;

    /**
     *
     * @param ApiManager $apiM
     */
    public function __construct(ApiManager $apiM)
    {
        $this->apiM = $apiM;
        $this->objectDbM = $this->apiM->container->get('object_database_manager');
    }

    /**
     * Returns a xml-string with allowed structure
     *
     * @return string
     */
    protected function _xmlAllowedElements()
    {
        return '<ObjectType required="true">'
            . '<Type/>'
            . '<Guid/>'
            . '<ReferenceId/>'
            . '<Offset/>'
            . '<Limit/>'
            . '<LanguageCode/>'
            . '</ObjectType>';
    }

    /**
     * Init model
     *
     * @param \SimpleXMLElement $simpleXML
     */
    public function init(\SimpleXMLElement $simpleXML)
    {
        $validator = new XmlValidator();
        if (!$validator->validate($simpleXML, $validator->strToXml($this->_xmlAllowedElements()))) {
            $this->apiM->errors()->addErrors($validator->errors()->getErrors());
            throw new InvalidRecordException('Invalid xml elements.');
        }

        // set helper model
        $this->helperModel = $this->apiM->serializer()->deserialize($simpleXML->ObjectType->saveXML(),
            'Sso\Webservices\ObjectBundle\Model\Request\Object\ShowMulti\Entity', 'xml');
        $this->validateEntity();

        $this->getObjectEntities();
    }

    public function getObjectEntities()
    {
        $offset = $this->helperModel->getOffset();
        $limit = $this->helperModel->getLimit();

        // Get objects by given type identifier - for example: get locations objects
        if (true == $this->getObjectEntitiesByTypeIdentifier($offset, $limit)) {
            return;
        }

        // Get objects by given reference id
        if (true == $this->getObjectEntitiesByReferenceId($this->helperModel->getReferenceId())) {
            return;
        }

        // Get objects by given guid
        if (true == $this->getObjectEntitiesByGuId($this->helperModel->getGuid())) {
            return;
        }

        $defaultCriteria = ['parent' => null];
        if ($offset !== null || $limit !== null) {
            $this->objectModels = $this->objectDbM->data()->object()->findBy($defaultCriteria, array(), $limit,
                $offset);
        } else {
            $this->objectModels = $this->objectDbM->data()->object()->findBy($defaultCriteria);
        }
    }

    /**
     * Get objects by given Guid
     *
     * @param string $guId
     * @return bool|null
     */
    private function getObjectEntitiesByGuId($guId)
    {
        if (!$guId) {
            return null;
        }

        $search = ['guid' => $guId];
        $this->objectModels = $this->objectDbM->data()->object()->findBy($search);
        return true;
    }

    /**
     * Get objects by given reference id
     *
     * @param string $referenceId
     * @return bool|null
     */
    private function getObjectEntitiesByReferenceId($referenceId)
    {
        if (empty($referenceId)) {
            return null;
        }

        $search = ['referenceId' => $referenceId];
        $this->objectModels = $this->objectDbM->data()->object()->findBy($search);
        return true;
    }

    /**
     * Get objects by given type identifier. For example: get location objects.
     *
     * @param int $offset
     * @param int $limit
     * @return bool|null
     */
    private function getObjectEntitiesByTypeIdentifier($offset, $limit)
    {
        $typeIdentifier = $this->helperModel->getTypeIdentifier();
        if (empty($typeIdentifier)) {
            return null;
        }

        // Get type model from type mapping
        $typeMapping = $this->getTypeMapping($typeIdentifier);

        $search = ['type' => $typeMapping->getType()];
        $this->objectModels = $this->objectDbM->data()->object()->findBy($search, array(), $limit, $offset);

        return true;
    }

    /**
     * @param string $typeIdentifier
     * @return EntityTypeMapping
     */
    private function getTypeMapping($typeIdentifier)
    {
        $typeMapping = $this->objectDbM->respository()->typeMapping()->find($typeIdentifier);

        if (!$typeMapping) {
            $this->apiM->errors()->addError(new ExternalError('o001', 'Type Mapping Identifier not found',
                'Type Mapping Identifier not found', 'Type Mapping Identifier not found'));
            throw new InvalidRecordException('Type Mapping Identifier not found');
        }

        return $typeMapping;
    }

    /**
     * @return int
     */
    public function getObjectsCount()
    {
        return $this->objectDbM->data()->object()->getCount();
    }

    /**
     *
     * @return EntityObject[]
     */
    public function getObjectModels()
    {
        return $this->objectModels;
    }

    /**
     *
     * @return Entity
     */
    public function getHelperModel()
    {
        return $this->helperModel;
    }

    /**
     * Validate entity
     *
     * @param array $validationGroups
     * @return bool
     */
    protected function validateEntity($validationGroups = null)
    {
        $errors = $this->apiM->validator()->validate($this->helperModel, $validationGroups);
        foreach ($errors as $error) {
            list($errorCode, $shortMessage, $longMessage) = explode('##', $error->getMessage());
            $this->apiM->errors()->addError(new ExternalError('v001', $errorCode, $shortMessage, $longMessage));
        }
        if (0 < count($errors)) {
            throw new InvalidRecordException('Invalid xml values');
        }
        return true;
    }
}
