<?php

/**
 * Class Delete
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\Webservices\ObjectBundle\Model\Request\ObjectValue\Delete;

use LifeStyle\Tools\WebserviceBundle\Error\Type\External as ExternalError;
use LifeStyle\Tools\WebserviceBundle\Validator\Xml as XmlValidator;
use LifeStyle\Tools\WebserviceBundle\Exception\InvalidRecordException;
use Sso\ApiBundle\Api\Manager as ApiManager;
use Sso\Webservices\ObjectBundle\Entity\ObjectAttributeValue as EntityObjectValue;
use Sso\Webservices\ObjectBundle\Model\Request\ObjectValue\Delete\Entity as ModelIn;
use Sso\Webservices\ObjectBundle\Database\Manager as ObjectDatabaseManager;

/**
 * Class Delete
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package    Sso\Webservices\ObjectBundle\Model\Request\ObjectValue\Delete
 */
class Delete
{

    /**
     * @var ApiManager
     */
    protected $apiM;

    /**
     * @var ObjectDatabaseManager
     */
    protected $objectDbM;

    /**
     * @var ModelIn
     */
    protected $helperModel;

    /**
     * @var EntityObjectValue
     */
    protected $objectValueEntity;

    /**
     *
     * @param ApiManager $apiM
     */
    public function __construct(ApiManager $apiM)
    {
        $this->apiM = $apiM;
        $this->objectDbM = $this->apiM->getContainer()->get('object_database_manager');
    }

    /**
     * Returns a xml-string with allowed structure
     *
     * @return string
     */
    protected function _xmlAllowedElements()
    {
        return '<ObjectValueType required="true">'
            . '<Id required="true"/>'
            . '</ObjectValueType>';
    }

    /**
     * Init model
     *
     * @param \SimpleXMLElement $simpleXML
     */
    public function init(\SimpleXMLElement $simpleXML)
    {
        $validator = new XmlValidator();
        if (!$validator->validate($simpleXML, $validator->strToXml($this->_xmlAllowedElements()))) {
            $this->apiM->errors()->addErrors($validator->errors()->getErrors());
            throw new InvalidRecordException('Invalid xml elements.');
        }

        // set helper model
        $this->helperModel = $this->apiM->serializer()->deserialize($simpleXML->ObjectValueType->saveXML(),
            'Sso\Webservices\ObjectBundle\Model\Request\ObjectValue\Delete\Entity', 'xml');
        $this->validateEntity();

        $this->objectValueEntity = $this->objectDbM->respository()->objectAttributeValue()->find($this->helperModel->getId());

        $this->deleteObject();
    }

    public function deleteObject()
    {
        if (!$this->objectValueEntity) {
            $this->apiM->errors()->addError(new ExternalError('ovd001', 'ObjectValue not found',
                'ObjectValue not found', 'ObjectValue not found'));
            throw new InvalidRecordException('ObjectValue not found');
        }

        $entityClone = clone $this->objectValueEntity;
        $this->objectDbM->data()->objectAttributeValue()->delete($this->objectValueEntity);

        $this->objectValueEntity = $entityClone;
    }

    /**
     * Validate entity
     *
     * @param array $validationGroups
     * @return bool
     */
    protected function validateEntity($validationGroups = null)
    {
        $errors = $this->apiM->validator()->validate($this->helperModel, $validationGroups);
        foreach ($errors as $error) {
            list($errorCode, $shortMessage, $longMessage) = explode('##', $error->getMessage());
            $this->apiM->errors()->addError(new ExternalError('v001', $errorCode, $shortMessage, $longMessage));
        }
        if (0 < count($errors)) {
            throw new InvalidRecordException('Invalid xml values');
        }
        return true;
    }

    /**
     *
     * @return EntityObjectValue
     */
    public function getObjectValueEntity()
    {
        return $this->objectValueEntity;
    }

    /**
     * @return ModelIn
     */
    public function getHelperModel()
    {
        return $this->helperModel;
    }

}
